/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: sfbufman.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *                          added prototypes to fix error:
 *                          error: static declaration of 'ahf_open' follows non-static declaration
 *                          #include <string.h>
 **************************************************************
** MANUAL
**	SFBUFMAN 3x "November 14, 1990"
**
** NAME
**	sfbufman - sfdu buffer management system
**
** SYNOPSIS
**
**	  #include   <sftools.h>
**
**	  int sfbufman(inbufinfo,rdinfo);
**
**	  struct  bufinfo  *inbufinfo;
**	  struct  rdinfo   *rdinfo;
**
**	  int sfread_file_in(rdinfo);
**
**	  struct rdinfo    *rdinfo;
**
** STRUCTURES
**
**	  structure bufinfo, bufsin, rdbuf used for buffer management
**
**	  struct  bufinfo {
**	     char  *buf;     points to buffer wanted.
**	     int   buflen;   length of buffer in bytes
**	     int   bufnumber; current buffer number.
**	     BOOL  contflag; TRUE for more buffers.
**	     int   bufrmode; returned mode for buffers.
**	  } ;
**
**	  bufrmode can be
**	  1. NO_NEW_BUFR - ignore this request.
**	  2. NEXT_BUFR - get next buffer.
**	  3. PREV_BUFR - get previous buffer.
**	  4. BOF - bottom of file get first buffer.
**	  5. EOF - get last buffer.
**
**	  struct rdinfo {
**	     SFFOPDEF(frd) - frd is file descriptor.
**	     char  *filename; file name of frd.
**	     char  **list_pntr; set of string pointers
**	                        used if filname = NULL
**	     int   indx_in_list - used along with list_pntr.
**	     char  *rdbuf; pointer to buffer struct.
**	     char  *start_struct_pntr; points to first structure used by 
**	                               buffers.
**	  }
**
**	  struct bufsin {
**	     int   tot_bufs;  total number of buffers.
**	     long  tot_bytes; total bytes in all buffers.
**	     int   bytes_per_buf; for allication of space.
**	     long  start_at_byte; number of bytes from start of file.
**	                          for first read.
**	     BOOL  more_reads; if All buffers full but
**	                       more reads or lists needed.
**	     struct  {
**	        char *pntr; points to buffer.
**	        int  lngth; number of bytes in buf.
**	        int  count; true buffer numbernot modulo MAXNUMBUFRS.
**	     } [MAXNUMBUFRS];
**	  }
**
** DESCRIPTION
**
**	'sfbufman' is 'buffer management' for files.
**	'sfbufman' contains two calls. One is 'sfbufman'
**	and the second is 'sfread_file_in'.
**
**	The areas, number of bytes etc, that a file, 
**	pointed to by 'rdinfo->frd', or strings pointed to
**	'rdinfo->list_pntr' are placed, is pointed to by 
**	'bufsin' which is the same as 'rdinfo->rdbuf'.
**	For this description of buffers the name 'rdbuf'
**	will be used for this structure.
**
**	Every 'rdbuf->bufs[i].pntr' must be set to NULL
**	upon entry to 'sfread_file_in' or the user must
**	supply the address of an area and a length for
**	'rdbuf->bufs[i].pntr' and 'rebuf->bufs[i].lngth'
**	where 'i' goes from 0 to MAXNUMBUFRS - 1.
**
**	If rdinfo->frd is not equal to FNULL then
**	'sfread_file_in' reads file (or portions of it) into
**	buffers pointed to by 'rdbuf' whose address of
**	structure is in 'rdinfo'.
** 	'sfread_file_in' reads the ith buffer from file
**	'rdinfo->frd' into 'rdbuf->bufs[i].pntr'. The number
**	of bytes read are placed in rdinfo->lngth. For
**	'sfread_file_in' 'rdbuf->bufs[i].count' is equal
**	to 'i'.
**
**	Else if rdinfo->frd is equal to FNULL then 'list_pntr'
**	is used. 'list_pntr[i]' points to the ith string
**	in the list. The last entry in 'list_pntrs' must
**	be a NULL pointer. 'sfread_file_in' fills up
**	each 'rdbuf->bufs[i].pntr' buffer from as much
**	as it can shuve into 'rdbuf->bufs[i].pntr' from
**	'list_pntr'.
**
**	If 'sfread_file_in' fills all of the buffers in
**	'rdbuf' then 'rdbuf->more_reads' is set TRUE
**	else it is set FALSE. 'rdbuf->tot_bufs' contains
**	total bufs read. 'rdbuf->tot_bytes' contains total
**	number of bytes
**
**
**	'sfbufman' performs various functions on the buffers
**	in 'rdbuf'. These are called out by 'bufinfo->bufrmode'
**	where 'bufinfo->bufrmode' can be
**
**	   1  NEXT_BUFR - This gets the next buffer from
**	      'rdbuf->bufs. If the next buffer is 'i+1' and
**	      has not been read into the buffers 'i+1' is
**	      set to modulo MAXNUMBUFRS but 'rdbuf->bufs[i+1]
**	      count is set to 'i+1'. This is a circular buffer.
**	  2  PREV_BUFR - This gets the previous buffer from
**	     'rdbuf->bufs'. If previous buffer is 'i-1' and
**	     k is modulo MAXNUMBUFRS of 'i-1' if
**	     'rdbuf->bufs[k]->count' equals 'i-1' then
**	     bufinfo is set to this buffer, else the file
**	     pointed to by 'rdinfo->frd' is read from the
**	     beggining until 'i-1' is read in. (the reason
**	     is that this needs to run on different types
**	     of machines and 'seek' does not always work).
**	  3  BOF - this goes to BOF, if necessary
**	     'sfread_file_in' is called.
**	  4  EOF - reads in final buffer. It does not
**	     point to end of buffer.
**	  5. USE_BUFINDX - the buffer number (count)
**	     'bufinfo->bufnumber' is used to position
**	     to the proper buffer.
**
** END MAN PAGES
*/
#endif   /* Header */
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"SCBUFMA0") csect(STATIC,"SCBUFMA1")
#endif

#include	<stdio.h>
#include	<stdlib.h>
#include    <string.h>
#include	<sftypes.h>
#include	<sffile.h>
#include	<sfbufmgr.h>

static	size_t	sizme;
static	int	read_lngth;

int
sfbufman(inbufinfo,rdinfo)

struct	bufinfo	*inbufinfo;
struct	rdinfo	*rdinfo;

{

int	i;
int	j;
int	k;
int	tot;
int	save_indx;
int	rt_code;
int	buflen;
struct	bufsin	*rdbuf;
char	*pntr;
int	len;

/*begin*/

rdbuf = (struct bufsin *)rdinfo->rdbuf;
read_lngth = rdbuf->bytes_per_buf;
/*curr_buf_indx says where we are*/
/*need next*/
switch (inbufinfo->bufrmode)
{
	case BOF: /*go to top of file*/
		inbufinfo->bufnumber = 0;
		if (rdbuf->bufs[0].count)
		{
			/*there has been a wrap around*/
			if (rdinfo->frd != SFFNULL)
			/*seek to start of file*/
				SFFSEEK(rdinfo->frd,0,SFSEEK_SET);
			rt_code = sfread_file_in(rdinfo);
			if (rt_code)
				return(rt_code);
		}
		inbufinfo->buf = rdbuf->bufs[inbufinfo->bufnumber].pntr;
		inbufinfo->buflen   = 
			rdbuf->bufs[inbufinfo->bufnumber].lngth;
		if (rdbuf->tot_bufs > 1)
			inbufinfo->contflag = TRUE;
		else
			inbufinfo->contflag = FALSE;
		return(inbufinfo->buflen);
	case PREV_BUFR:
		inbufinfo->bufnumber--;
	break;
	case NEXT_BUFR:
		inbufinfo->bufnumber++;
	break;
	case USE_BUFINDX:
	break;
	case EOF: /*End of file*/
		if (!rdbuf->more_reads)
		{
			i = rdbuf->tot_bufs;
			inbufinfo->bufnumber = i;
			inbufinfo->buf = rdbuf->bufs[i].pntr;
			inbufinfo->contflag = FALSE;
			inbufinfo->buflen = rdbuf->bufs[i].lngth;
			return(rdbuf->bufs[i].lngth);
		}
		/*read until end*/
		j = rdbuf->tot_bufs;
		for (;;)
		{
			k = j%MAXNUMBUFRS;
			rdbuf->bufs[k].lngth = SFFREAD(
					rdinfo->frd,
					rdbuf->bufs[k].pntr,
					read_lngth);
			if (!rdbuf->bufs[k].lngth)
			{
				free(rdbuf->bufs[k].pntr);
				i = j - 1;
				rdbuf->more_reads = FALSE;
				/*i points to one needed.*/
				inbufinfo->bufnumber = i;
				k = i%MAXNUMBUFRS;
				inbufinfo->contflag = FALSE;
				inbufinfo->buf = rdbuf->bufs[k].pntr;
				inbufinfo->buflen = rdbuf->bufs[k].lngth;
				return(rdbuf->bufs[k].lngth);
			}
			j++;
		}
	default:
	break;
}

for (;;)
{
	i = inbufinfo->bufnumber%MAXNUMBUFRS;
	if (inbufinfo->bufnumber >= rdbuf->tot_bufs)
	{
		if (!rdbuf->more_reads)
		{
			return(0);
		}
		tot = rdbuf->tot_bufs;
		for ( j = tot ; j <= inbufinfo->bufnumber ; j++)
		{
			k = j%MAXNUMBUFRS;
			if (rdinfo->frd == SFFNULL)
			{
				/*get from buffer*/
				if (
				  rdinfo->list_pntr[rdinfo->indx_in_list]
				    == NULL)
				{
					rdbuf->bufs[i].lngth = 0;
					rdbuf->bufs[i].pntr = NULL;
					rdbuf->more_reads = FALSE;
				}
				else
				{
					i = k;
					pntr = rdbuf->bufs[i].pntr;
					k = rdinfo->indx_in_list;
					for (;;)
					{
						if (rdinfo->list_pntr[k]
						    == NULL)
						{
							rdbuf->more_reads =
								FALSE;
							break;
						}
						len = strlen(pntr);
						if ((len + strlen(
					(char *)rdinfo->list_pntr)) >
							(int)sizme)
						{
							break;
						}
						sprintf(&pntr[len],
							"%s",
						  rdinfo->list_pntr[k]);
						rdinfo->indx_in_list++;
						k = rdinfo->indx_in_list;
					}
					rdbuf->bufs[i].lngth = len;
					rdbuf->tot_bufs++;
					rdbuf->tot_bytes += len;
				}
			}
			else
			{
				/*get from file*/
				rdbuf->bufs[k].lngth = SFFREAD(
						rdinfo->frd,
						rdbuf->bufs[k].pntr,
						read_lngth);
				if (rdbuf->bufs[k].lngth < 0)
				{
					return(-1);
				}
				rdbuf->bufs[i].count = j;
				rdbuf->tot_bufs++;
				rdbuf->tot_bytes += rdbuf->bufs[k].lngth;
				if (rdbuf->bufs[i].lngth < read_lngth)
				{
					rdbuf->more_reads = FALSE;
				}
			}
		}
	}
	else
	{
		if (rdbuf->bufs[i].count > inbufinfo->bufnumber)
		{
			/*we have counted back more then we can get*/
			if (rdinfo->frd != SFFNULL)
				/*seek to start of file*/		
				SFFSEEK(rdinfo->frd,0,SFSEEK_SET); 
			rt_code = sfread_file_in(rdinfo);
			if (rt_code)
				return(rt_code);
			continue;
		}
	}
	break;
}
if (rdbuf->more_reads  ||
    inbufinfo->bufnumber+1 != rdbuf->tot_bufs)
	inbufinfo->contflag = TRUE;
else
	inbufinfo->contflag = FALSE;
inbufinfo->buf = rdbuf->bufs[i].pntr;
inbufinfo->buflen   = rdbuf->bufs[i].lngth;
if (!inbufinfo->buflen)
	inbufinfo->contflag = FALSE;
return(inbufinfo->buflen);
}
int
sfread_file_in(rdinfo)

struct	rdinfo	*rdinfo;

{

int	rt_code;
int	i;
int	j;
struct	bufsin	*rdbuf;
char	*pntr;
int	len;
int	rdlen;

/*begin*/
rdbuf = (struct bufsin *)rdinfo->rdbuf;
/*findfile length*/
rdbuf->more_reads = TRUE;
sizme = rdbuf->bytes_per_buf;
rdbuf->tot_bufs  = 0;
rdbuf->tot_bytes = 0;
read_lngth = rdbuf->bytes_per_buf;
if (rdinfo->frd == SFFNULL)
{
	/*use buffers*/
	rdinfo->indx_in_list = 0;
	for ( i = 0 ; i  < MAXNUMBUFRS ; i++)
	{
		if (rdinfo->list_pntr[rdinfo->indx_in_list] == NULL)
		{
			rdbuf->more_reads = FALSE;
			return(0);
		}
		if (rdbuf->bufs[i].pntr == NULL)
		{
			/*compute size for malloc*/
			sizme = 0;
			j = rdinfo->indx_in_list;
			for (;;)
			{
				if (rdinfo->list_pntr[j] == NULL)
					break;
				sizme += 
				   (U_INT)strlen(rdinfo->list_pntr[j]);
				if (sizme >= rdbuf->bytes_per_buf)
					break;
				j++;
			}
			rdbuf->bufs[i].pntr = malloc(sizme);
			if (rdbuf->bufs[i].pntr == NULL)
			{
				printf(
				 "Error in sfread_file_in for lists");
				return(-1);
			}
		}
		pntr = rdbuf->bufs[i].pntr;
		pntr[0] = '\0';
		for (;;)
		{
			len = strlen(pntr);
			if (rdinfo->list_pntr[rdinfo->indx_in_list] 
				== NULL)
			{
				rdbuf->more_reads = FALSE;
				break;
			}
			if ((len + strlen(rdinfo->list_pntr[
				rdinfo->indx_in_list]))
			    > (int)sizme)
				break;
			sprintf(&pntr[len],
				"%s",
				rdinfo->list_pntr[rdinfo->indx_in_list]
				);
			rdinfo->indx_in_list++; /*next in line*/
		}
		rdbuf->bufs[i].lngth = len;
		rdbuf->tot_bytes += len;
		rdbuf->tot_bufs++;
		rdbuf->bufs[i].count = i;
	}
	return(0);
}
/*this is file country*/
/*seek to start of file*/
SFFSEEK(rdinfo->frd,0,SFSEEK_SET);
for ( i = 0 ; i < MAXNUMBUFRS ; i++)
{
	if (rdbuf->bufs[i].pntr == NULL)
	{
		rdbuf->bufs[i].pntr = malloc(sizme);
		if (rdbuf->bufs[i].pntr == NULL)
		{
			printf(
				"Error in sfread_file_in for files");
			return(-1);
		}
	}
	if (!i   &&  rdbuf->start_at_byte)
	{
		/*This means to read down 'start_at_byte' bytes*/
		len = rdbuf->start_at_byte;
		if (!rdbuf->bufs[i].lngth)
		{
			rdbuf->bufs[i].lngth = read_lngth;
		}
		while (len)
		{
			if (len >= rdbuf->bufs[i].lngth)
			{
				rdlen = rdbuf->bufs[i].lngth;		
			}
			else
			{
				rdlen = len;
			}
			j = SFFREAD(rdinfo->frd,
				     rdbuf->bufs[i].pntr,
				     rdlen);
			if (j != rdlen)
			{
				printf(
				"Error in read. 'start_at_byte'");
				return(-1);
			}
			len -= j;
		}
	}		
	rdbuf->bufs[i].lngth = SFFREAD(rdinfo->frd,
				     rdbuf->bufs[i].pntr,
				     read_lngth);
	if (!rdbuf->bufs[i].lngth)
	{
		free(rdbuf->bufs[i].pntr);
		rdbuf->bufs[i].pntr = NULL;
		rdbuf->more_reads = FALSE;
		break;
	}
	rdbuf->bufs[i].count = i;
	rdbuf->tot_bufs++;
	rdbuf->tot_bytes += rdbuf->bufs[i].lngth;
	if (rdbuf->bufs[i].lngth < read_lngth)
	{
		rdbuf->more_reads = FALSE;
		break;
	}
}
return(0);
}
