/*
         1         2         3         4         5         6         7
123456789012345678901234567890123456789012345678901234567890123456789012
*/
#ifndef Header
/*****************************************************************
 * TITLE: scet2clk.c
 *
 * AUTHOR:  Unknown
 *          Aug 31, 1994
 *
 * MODIFIED:    Ray Bambery
 *          Aug 24, 2020 -  removed label after #ENDIF Header 
 *                          add prototypes 
 ******************************************************************
** MANUAL
**	SCET2CLK 3x, "December 7,1993"
**
** NAME
**	scet2clk - Find SCLK from an ASCII coefficient file.
**
** SYNOPSIS
**
**	  #include <scet2clk.h>
**	
**	  int scet2clk(option,struct_pntr);
**	             
**
**	  char  *option;
**	  char	*struct_pntr;
**
** PARAMETERS
**
**	  1. option - points to string with 'OPEN',
**	     'SCLK', 'CLOSE'
**	  2. struct_pntr - points to structure to use for
**	     the option type.
**
** STRUCTURES
**	
**	  struct coefopen
**	  {
**	        SFFOPDEF(cffd);
**	       	char  filename[100];
**	  };
**
**	  typedef struct coefopen CFOPEN;
**
**	  struct coefclose
**	  {
**	       	SFFOPDEF(cffd);
**	  };
**
**	  typedef struct coefclose CFCLOSE;
**
**	  struct  coefsclk
**	  {
**	        SFFOPDEF(cffd);
**	        DBL   scet;
**	        DBL   sclk;
**	        char  time_type[12];
**	        int   time_indx;
**	  };
**
**	  typedef struct coefsclk CFCLK;
**
** DESCRIPTION
**
**	'scet2clk' opens a Ulysses ASCII coefficient file, finds the 
**	start of the coefficients. Then given a doubl precision time
**	from some epoch (1950 or 2000) find the corresponding
**	SCLK.
**
** EXAMPLE


EXAMPLE 1.
	Ulyses SCLK/GLL coefficient file (UNYSIS)
	
$$ULS       SCLK/SCET COEFFICIENT FILE                                         1
*SCID       ULS                                                                2
*FILE       UDMT*SCLKSCETCOEF.                                                 3
*CREATION   90-285/14:49:10                                                    4
*UPDATE     93-106/10:44:39                                                    5
*RATE        2.000030997                                                       6
*PART 1     90-279/19:10:49.727 @      100                                     7
*                                                                             50
*           SCET(UTC) = A0 + A1*(SCLK - SCLK0)                                51
*                                                                             52
*    SCLK0             A0            DUT       A1           ENTRY       ENTRY 53
* EFFECT. SCLK  EFFECT. SCET(UTC)  ET-UTC EFFECT. RATE      EPOCH        SEQ  54
*SCLKCOUNT.000 YY-DDD/HH:MM:SS.FFF  SECS   SECONDS/RIM    LOCAL TIME     NUM  55
$$EOH                                                                         99
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22       100

EXAMPLE 2.
CCSD3ZS00001$$sclk$$NJPL3KS0L015$$scet$$
MISSION_NAME=ULYSSES;
SPACECRAFT_NAME=ULS;
DATA_SET_ID=SCLK_SCET;
FILE_NAME=SCLK_SCET.00001;
PRODUCT_CREATION_TIME=1990-285T14:49:10;
PRODUCT_VERSION_ID=1;
PRODUCER_ID=DMT;
APPLICABLE_START_TIME=1990-285T14:49:10;
APPLICABLE_STOP_TIME=1999-12-30T12:00:00.000;
MISSION_ID=3;
SPACECRAFT_ID=55;
CCSD3RE00000$$scet$$NJPL3IS00613$$data$$
*____SCLK0_______  ____SCET0____________ _DUT__ _SCLKRATE___ 
          0.000.0  1990-279T19:10:49.727 57.184 1.994697308
       1000.000.0  1990-279T19:44:04.424 57.184 2.000052742
 
*/
#endif /*  Header	*/
#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"SCET2CL0") csect(STATIC,"SCET2CL1")
#endif

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<glbltme.h>
#include	<scet2clk.h>

#if (IBM_MAIN_FRAME)
/*place pragma to allow for entry points for JTPM*/
#pragma csect(CODE,"SCET2CL0") csect(STATIC,"SCET2CL1")
#endif
#if (NOPROTO)
extern	void	dictmcnv();
extern	void	dictmcnv();
extern	int	cnvs2dbl();
#else
extern	void	dictmcnv(long int *,char *,char *,DBL *,char *,
			  char *,char *,char *,char *);
extern	void	diclkcnv(long int *,char *,DBL *,char *,char *,
			  char *,char *,char *);
extern	int	cnvs2dbl(char *,char *);

int scet2clk(char *option,char *struct_pntr);
static  coef_open(void);
static  coef_fndclk(void);
static findslot(void);
int coef_close(void);

#endif

static	CFOPEN	*CFO;
static	CFCLOSE	*CFC;
static	CFSCLK	*CFS;
static	char	buf[181];
static	long	int	clkoffset;
static	DBL	prev_scet;
static	DBL	base_sclk;
static	DBL	next_scet;
static	DBL	A1;
static	int	rdlen;
static	int	toread;
static	int	rec_len;
static	long	int	tarray[SIZE_OF_TARRAY];
static	DBL	dpsecs[SIZE_OF_DPS];
static	char	mon[4];
static	char	typein[12];
static	char	typeout[12];
static	long	int	carray[SIZE_OF_CARRAY];
static	char	rtcode[5];
static	BOOL	use_bufa;
static	char	*bufa;
static	char	*bufb;

/*************************************************************/	
int 
scet2clk(option,struct_pntr)

char	*option;
char	*struct_pntr;

{

int	rtrn;
char	*pntr;

if (strcmp(times_data_init,"YES"))
{
	ditminit();
}
/*get opiton*/
switch (option[0])
{
	case 'O':
	case 'o': /*open*/
		CFO = (CFOPEN *)struct_pntr;
		rtrn = coef_open();
	break;
	case 'C':
	case 'c':
		CFC = (CFCLOSE *)struct_pntr;
		rtrn = coef_close();
	break;
	case 'S':
	case 's':
		CFS = (CFSCLK *)struct_pntr;
		rtrn = coef_fndclk();
	break;
	default:
		fprintf(stderr,
			"%s is an unknown option\n",
			option);
		return(-1);
}

return(rtrn);
}
/********************************************************************/
static
coef_open()

{

char	*pntr;
BOOL	is_first;
long	int	len;

/*open files for sceclk in*/
SFFOPEN(CFO->cffd,CFO->filename,SFRDONLY);
if (SFFERROPEN(CFO->cffd))
{
	fprintf(stderr,
		"Error in opening %s\n",
		CFO->filename);
	return(-1);
}
/*set to start of records here*/
/* read until we find $$EOH*/
toread = 180;
is_first = TRUE;
buf[180] = '\0';
clkoffset = 0;
for (;;)
{
	rdlen = SFFREAD(CFO->cffd,buf,toread);
	if (rdlen <= 0)
	{
		fprintf(stderr,
			"Error in reading in header\n");
		return(-1);
	}
	if (is_first)
	{
		is_first = FALSE;
		/*chek for $$*/
		if (strncmp(buf,"$$ULS",5))
		{
			fprintf(stderr,
			 "File %s is not a ULS coefficient file\n",
			 CFO->filename);
			return(-1);
		}
		/*find rec_len*/
		pntr = strchr(buf,'*');
		if (pntr == NULL)
		{
			fprintf(stderr,
			 "File %s is not a ULS coefficient file\n",
			 CFO->filename);
			return(-1);
		}
		rec_len = (int)((long int)pntr - (long int)buf);
		clkoffset += (long int)rdlen;
		continue;
	}
	pntr = strchr(buf,'$');
	if (pntr != NULL)
	{
		if (strncmp(pntr,"$$EOH",5))
		{
			clkoffset += (long int)rdlen;
			continue;
		}
		/*found it find next*/
		len = (long int)pntr - (long int)buf;
		clkoffset += (long int)rec_len + len;
		break;
	}
	clkoffset += (long int)rdlen;
}
SFFSEEK(CFO->cffd,clkoffset,SFSEEK_SET);
prev_scet = 0;
base_sclk = 0;
next_scet = 0;
A1 = 0;
use_bufa = TRUE;
bufa = buf;
bufb = &buf[rec_len+1];
return(0);
}
/**************************************************************************/
static
coef_fndclk()

{

int	rtrn;
DBL	delta_scet;

/*************************************************************
          1         2         3         4         5         6
01234567890123456789012345678901234567890123456789012345678901234567890
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22       100
**********************************************************/
if (CFS->eod)
{
	/*compute SCLK from scet etc*/
	CFS->sclk = base_sclk + (CFS->scet - prev_scet)/A1;
	return(1);
}
if (CFS->scet > next_scet)
{
	rtrn = findslot();
	if (rtrn < 0)
	{
		return(rtrn);
	}
}
/*compute SCLK from scet etc*/
CFS->sclk = base_sclk + (CFS->scet - prev_scet)/A1;
return(0);
}	
/***************************************************************************/
static
findslot()

{

char	*pntr;
BOOL	next_bufa;
int	rtrn;

strcpy(typein,"CHAR");
/*type out is in CFS->time_type i.e. DP50 or DP2000*/
for (;;)
{
	/*set up double buffering*/
	if(use_bufa)
	{
		pntr = bufa;
		use_bufa = FALSE;
		next_bufa = TRUE;
	}
	else
	{
		pntr = bufb;
		use_bufa = TRUE;
		next_bufa = FALSE;
	}
	rdlen = SFFREAD(CFS->cffd,pntr,rec_len);
	if (rdlen < 0)
	{
		fprintf(stderr,
		 "Error in readin record from coefficient file.\n");
		 return(-1);
	}
	if (!rdlen  ||  pntr[0] == '$')
	{
		CFS->eod = TRUE;
		return(1);
	}
	/*find what ever		
*SCLKCOUNT.000 YY-DDD/HH:MM:SS.FFF  SECS   SECONDS/RIM    LOCAL TIME     NUM  55
          1         2         3         4         5         6         7 
01234567890123456789012345678901234567890123456789012345678901234567890                                                                        99
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22
 point to start of time place '\0' in end of time*/
 	pntr[34] = '\0'; /*make a string*/
	dictmcnv(tarray,mon,NULL,dpsecs,&pntr[15],typein,
		NULL,NULL,rtcode);
	if (rtcode[3] == 'X')
	{
		/*error*/
		fprintf(stderr,
			"Error in time for %s err code is %s\n",
			&pntr[15],rtcode);
		return(-1);
	}
	if (CFS->scet < dpsecs[CFS->time_indx])
	{
		/*we found it*/
		break;
	}
	prev_scet = dpsecs[CFS->time_indx];
}
/*ok now convert previous buffer if next_bufa is TRUE then prev is bufb 
  and vice versa
*SCLKCOUNT.000 YY-DDD/HH:MM:SS.FFF  SECS   SECONDS/RIM    LOCAL TIME     NUM  55
          1         2         3         4         5         6         7 
01234567890123456789012345678901234567890123456789012345678901234567890                                                                        99
         0.000 90-279/19:10:49.727 57.184  1.994697308 93-026/11:31:22

ok call dictmcnv*/	
/*compute clk from various clk formats from tstrng*/
next_scet = dpsecs[CFS->time_indx];
if (next_bufa)
{
	pntr = bufb;
}
else
{
	pntr = bufa;
}
pntr[14] = '\0';
diclkcnv(carray,"ULS",&base_sclk,&pntr[1],typein,NULL,NULL,rtcode);
/*get A1*/
pntr[54] = '\0';
rtrn = cnvs2dbl(&pntr[42],(char *)&A1);
if (rtrn < 0)
{
	fprintf(stderr,
		"Error in converting %s to double precision.\n",
		&pntr[42]);
	return(-1);
}
return(0);
}
/********************************************************************************/
coef_close()

{

SFFCLOSE(CFC->cffd);

return(0);

}
