/***********************************************************************
TITLE:	sfdu library for reading SFDU ancillary data files
CIN: sfdu.c,v 1.1.1.1 1994/08/31 19:22:02 root Exp

FILE NAME:	sfdu.c
AUTHOR:		David Wagner
CREATED: 	June 16, 1989
LAST UPDATE: 	1994/08/31 19:22:02
            2021/2/25
            Ray Bambery - added printf statements
            to accompany sprintf statments for my logging
            2021/06/07 
            Ray Bambery - added printf ("sfdu_fgets: length = %d\n",length);
            to help debug abort in reading SCLK_SCET.31 or SCLK_SCET.32
            when vgr_sfdu2edr.C is not in debug_on=1
LAST UPDATE:    Dec 07, 2021 - Ray Bambery
VERSION:    1.6
COMPILER:   gcc 7.2 on Centos 7 (pistol.jpl.nasa.gov)
OS:         Solaris 10.2 and Centos 7 
        
REFERENCES: 
WARNINGS:
BUGS:
UNIT TEST:	cc -DUNIT_TEST -o test
WAIVERS:

CHANGE HISTORY:
	sfdu.c,v
 * Revision 1.6 Centos 7/Solaris 10.2 - Ray Bambery
 * add include <stdlib.h> for atol - msg:  
 * warning: implicit declaration of function ‘atol’

 * Revision 1.1.1.1  1994/08/31  19:22:02  root
 * Initial Release
 *
 * Revision 1.5  1992/12/31  04:17:12  kevin
 * corrected RCS keywords
 *
 * Revision 1.4  1991/08/05  21:41:57  dsfa
 * changed warning message
 *
 * Revision 1.3  1991/04/03  22:44:22  dsfa
 * changed Invalid SFDU message to a warning
 *
 * Revision 1.2  1991/03/28  22:45:44  dsfa
 * corrected debug MSG warnings
 *
 * Revision 1.1  1991/03/13  17:59:31  dsfa
 * Initial revision
 *
***********************************************************************/

//    Routine                   called by
//    find_end_marker
//    sfdu_open                 vgr_scet, load_file
//    sfdu_next_sfdo            read_sclk
//    sfdu_next_sub_sfdu        
//    sfdu_sfdo_read            read_sclk
//    sfdu_fgets                read_sclk
//    sfdu_close                vgr_scet, read_sclk, load_file
//    main  TEST_PROGRAM


#ifdef Header
/*
** MANUAL
**	time_sfdu 3x "1.1.1.1"
** NAME
**	time_sfdu -- library functions for manipulating SFDU files
** SYNOPSIS
**	#include <sfdu.h>
**
**	SFDU *	sfdu_open(filename, mode)
**		char * filename;
**		char * mode;
**
**	SFDU_CLASS	sfdu_next_sfdo(sfdu, sfdo)
**		SFDU * sfdu;
**		SFDO * sfdo;
**
**	SFDU_CLASS	sfdu_next_sub_sfdo(parent_sfdo, sfdo)
**		SFDO * parent_sfdo;
**		SFDO * sfdo;
**
**	ulong	sfdu_sfdo_read(buffer, size, count, sfdo)
**		char * buffer;
**		int size, count;
**		SFDO * sfdo;
**
**	char *	sfdu_fgets(buffer, length, sfdo)
**		char * buffer;
**		int length;
**		SFDO * sfdo;
**
**	int	sfdu_close(sfdu)
**		SFDU * sfdu;
**
**	The following macros are also available for accessing SFDO
**	or SFDU structure information:
**
**	char *	sfdu_label(SFDU * -or- SFDO *)
**
**	char *	sfdu_version(SFDU * -or- SFDO *)
**
**	SFDU_CLASS sfdu_class(SFDU * -or- SFDO *)
**
**	ulong	sfdu_len(SFDU * -or- SFDO *)
**
**	ulong	sfdu_unread(SFDU * -or- SFDO *)
**
**	char *	sfdu_ddpid(SFDU * sfdu)
**
**	FILE *  sfdu_filep(SFDU *)
**
**	FILE *	sfdo_filep(SFDO *)
**
**	SFDU *	sfdo_parent(SFDO *)
**
**	where:
**		ulong is defined as unsigned long.
**		SFDU and SFDO stuctures are defined in time_sfdu.h.
**
** DESCRIPTION
**	This library of routines provides a standard method to read SFDU
**	encapsulated data files along the lines of stream i/o in the C
**	stdio library.
**
**	__ sfdu_open() is equivalent to __ fopen() in that it takes a physical
**	file name and attempts to open it as an SFDU.  An SFDU structure is
**	created and the label data of the SFDU is read and parsed into
**	this structure.  A pointer to this structure is returned on a
**	successful open.  __ sfdu_close() is used to close an SFDU file
**	in the same manner as __ fclose().
**
**	__ sfdu_next_sfdo() and __ sfdu_next_sub_sfdo() are used to open
**	data objects contained in the SFDU. These functions are called
**	recursively as sub-data objects are found.  Once a data object is
**	open, then either __ sfdu_sfdo_read() or __ sfdu_fgets()
**	may be called to read the data portion of the data object.  Both or
**	these functions will return SFDU_EOF (0) upon reaching the end of
**	the data portion of the data object.  __ sfdu_sfdo_read() behaves like
**	__ fread(), and __ sfdu_fgets() behaves like 
**	__ fgets().
**
**	A set of macros are provided to allow access to the data stored in
**	the SFDU and SFDO data structures.  It is strongly recommended
**	that programmers use these macros instead of directly accessing
**	the structures since there is no guarantee that the structures
**	will remain constant.  In future versions, some of these macros
**	may be replaced with functions.
** INPUTS
**	__ SFDU * is the pointer returned by __ sfdu_open().
**	__ SFDO * is a pointer to the SFDO object returned by
**	__ sfdu_next_sfdo() and __ sfdu_next_sub_sfdo().  These
**	functions do now allocate space for the SFDO structure, so the
**	calling program should do so.  This is done this way since
**	you will usually only need one SFDO structure which will be
**	reused as ned objects are read and discarded.
** OUTPUTS
**	SFDU_CLASS is the class byte from the SFDO label of the object
**	being read.  The following defines are used:
**
** __VALUES__
**	  #define SFDUCID_CATALOG	'K'	// Catalog Data 
**	  #define SFDUCID_INFO		'I'	// Main Data 
**	  #define SFDUCID_SUPPL		'S'	// Supplementary Data 
**	  #define SFDUCID_TEXT		'T'	// Text 
**	  #define SFDUCID_DDP		'D'	// Data Definition Package 
**	  #define SFDUCID_DEDICT	'E'	// Data Entity Dictionary 
**	  #define SFDUCID_DFDESC	'F'	// Data Format Description 
**	  #define SFDUCID_DLANG		'L'	// Language Description 
**	  #define SFDUCID_OBJECT	'C'	// Object Identification 
**	  #define SFDUCID_AGR		'R'	// Reference/Aggregation Marker 
**	  #define SFDUCID_VOLUME	'V'	// Physical Volume Info 
**	  #define SFDUCID_UNCLAGR	'Z'	// Unclassified Aggregation 
**
** RETURN VALUE
**
** WARNINGS
**	These routines will get confused if files are not transferred in
**	binary mode (i.e., if carriage return characters, or other control
**	characters get deleted).
** BUGS
**	Currently, the SFDO-reading functions use a simple-minded search
**	method to find the end of delimited-by-marker blocks.  Thus, they
**	might not find the end of such blocks and return junk as data.
**
**	The current version only includes SFDU-reading capabilities. 
**	SFDUs cannot be opened in any mode other than "r".  Later releases
**	should include write options.
**
**	Only the delimiter types 'A', 'S', and 'F' are currently implemented.
** AUTHOR
**	David Wagner, Jet Propulsion Laboratory
** SEE ALSO
*/
#endif    //Header


#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>    /* for atol */
#include "sfdu.h"

/* This macro returns true if the char is in the string s
** where s is supposed to be a set.
*/
#define IN_SET(c, s) (strchr(s, c) != NULL)

//----------------------------------------------------------------------
/* find the Version-3 SFDU label containing the end marker label
** given in the second argument.  An end marker label is a CCSD3RE00000
** type label where the last 8-octets are the same as the marker
** string.
*/
static long
find_end_marker(fp, marker)
	FILE * fp;
	char * marker;
{
	char label[21];
	long startpos, markpos;
	char * expect = label;
	char c;
	int state = 0;

	/* save initial position */
	startpos = ftell(fp);

	/* construct the label to look for */
	strcpy(label, "CCSD3RE00000");
	strncat(label, marker, 8);
/*  ~
	DEBUG_PRN("DEBUG: Searching for marker label = %s\n", label);
	DEBUG_PRN("DEBUG: Starting position = %ld\n", startpos);
*/
	for (state = 0; !feof(fp); )
	{
		if ((c = getc(fp)) == label[state])
		{
			/* check for pattern completion */
			if (++state >= 20)
			{
				markpos = ftell(fp) - state;
				fseek(fp, startpos, SEEK_SET);
				return(markpos);
			}
		} else {
			if (state > 0)
			{
				/* back up by state */
				fseek(fp, -state, SEEK_CUR);
				state = 0;
			}
		}
	}
	DEBUG_PRS("DEBUG: Failed to find pattern!!");
	return(-1L);	/* error -- not found */
}
//------------------------------------------------------------------------
SFDU *
sfdu_open(filename, mode)
char * filename;
char * mode;
{
	SFDU * sfdu;
	char buff[SFDU_HEADER_LEN+1];
	long start_pos;

	/* malloc a struct SFDU */
	if ((sfdu = (SFDU *)malloc(sizeof(SFDU))) == NULL)
	{
		/* error -- can't malloc */
		fprintf(stderr, "failed to malloc an SFDU\n");
        printf ("sfdu_open: failed to malloc an SFDU\n");  /* RJB */
		return(NULL);
	}

	/* try to open the file */
    	if ((sfdu->fd = fopen(filename, "rb")) == NULL)
    	{
    		/* error -- can't open the file */
		fprintf(stderr, "cannot open %s\n", filename);
        printf("sfdu_open: cannot open %s\n", filename);   /* RJB */
    		free(sfdu);
    		return(NULL);
    	}

	/* parse the header and length fields */
    	if ((int) fread(buff, 1, SFDU_HEADER_LEN, sfdu->fd) < SFDU_HEADER_LEN)
    	{
    		/* error -- unexpected EOF */
		fprintf(stderr, "Unexpected EOF: incomplete SFDU\n");
        printf("sfdu_open: Unexpected EOF: incomplete SFDU\n");   /* RJB */
    		free(sfdu);
    		return(NULL);
    	}
	strncpy(sfdu->header, buff, SFDU_HEADER_LEN);
	sfdu->header[SFDU_HEADER_LEN] = CHARNULL;
	/* primary label must be 'CCSD...' or it ain't an SFDU*/ 
//printf("===sfdu->header = %s \n", sfdu->header);   //removed 5-17-2021 - RJB
     /*   */
	if (strncmp(sfdu->header, "CCSD", 4) != 0)
	{
		fprintf(stderr, "Warning: Invalid SFDU label in file %s\n", filename);
		printf("sfdu_open: Warning: Invalid SFDU label in file %s\n", filename);
    		free(sfdu);
    		return(NULL);
	}
	sfdu->class = buff[SFDU_CLASS_POS];
	/* if class isn't Z or I then something's wrong */
	if (!IN_SET(sfdu->class, SFDU_AGR_CLASSES))
	{
		fprintf(stderr, "Invalid SFDU class\n");
        printf ("sfdu_open: Class %s - Invalid SFDU class\n"); /* RJB */
    		free(sfdu);
    		return(NULL);
	}
	strncpy(sfdu->ddpid, &buff[SFDU_DDPID_POS], SFDU_DDPID_LEN);
	sfdu->ddpid[SFDU_DDPID_LEN] = CHARNULL;

	sfdu->version = buff[SFDU_VERSION_POS];

	if (sfdu->version == '3')
	{
/* ~
		DEBUG_PRS("DEBUG Reading type 3 label");
*/
		/* determine delimiter type.  If not delimited by length
		** then we will scan ahead to find the marker and set
		** the length field appropriately.
		*/
		start_pos = ftell(sfdu->fd);
		switch (buff[SFDU_LTYPE_POS])
		{
		case 'A' :
			/* read length directly */
			sfdu->len = atol( &(sfdu->header[SFDU_LEN_POS]) );
			break;
		case 'S' :
			/* This function searches ahead in the file for the
			** marker label containing the given marker
			** string.  On return, the fp is set at the
			** beginning of the SFDU label containing the
			** marker.  The function returns the fseek value
			** at that position.
			*/
			sfdu->len = (ulong)(find_end_marker(sfdu->fd, 
					&(sfdu->header[SFDU_LEN_POS]) )
					- start_pos);
/* ~
			DEBUG_PRN("DEBUG: Object length = %lu\n", sfdu->len);
*/
			break;
		case 'F' :
			/* save file position, then seek ahead to EOF.
			** set length to EOF posn - start_pos.
			*/
			fseek(sfdu->fd, 0, SEEK_END);
			sfdu->len = ftell(sfdu->fd) - start_pos;
			fseek(sfdu->fd, start_pos, SEEK_SET);
			break;
		default:
			break;
		}
	}
	sfdu->unread = sfdu->len;

    	return(sfdu);
}
//--------------------------------------------------------------------------
SFDU_CLASS
sfdu_next_sfdo(sfdu, sfdo)
SFDU * sfdu;
SFDO * sfdo;
{
	char buff[SFDU_HEADER_LEN+1];
	long start_pos;

	/* initialize the sfdo by copying the sfdu pointer into it */
	sfdo->sfd = sfdu;

	do {
		/* read and validate header */
		if ((int)fread(buff, 1, SFDU_HEADER_LEN, sfdu->fd) < SFDU_HEADER_LEN)
		{
			DEBUG_PRS("DEBUG |EOF|");
			return(SFDU_ERROR);
		}
	/* make sure label is the start of an SFDO -- could be an end 
	** marker label (CCSD*RE) in which case we should skip it
	*/
    /* fill in the SFDO data structure - sfdu.h */
	}
	while ( strncmp(buff,"CCSD",4)==0 && strncmp(&buff[5],"RE",2)==0 );

	strncpy(sfdo->header, buff, SFDU_HEADER_LEN);
	sfdo->header[SFDU_HEADER_LEN] = CHARNULL;
	sfdo->class = buff[SFDU_CLASS_POS];
	strncpy(sfdo->ddpid, &buff[SFDU_DDPID_POS], SFDU_DDPID_LEN);
	sfdo->ddpid[SFDU_DDPID_LEN] = CHARNULL;
/* ~
	DEBUG_PRN("DEBUG: SFDO: Read Header: %s\n", sfdo->header);
*/
	sfdo->startpos = start_pos = ftell(sfdo_filep(sfdo));
	sfdo->version = buff[SFDU_VERSION_POS];
	if (sfdu->version == '3')
	{
		/* determine delimiter type.  If not delimited by length
		** then we will scan ahead to find the marker and set
		** the length field appropriately.
		*/
/* ~
		DEBUG_PRN("DEBUG: Type 3 pos = %ld\n", start_pos);
*/
		switch (buff[SFDU_LTYPE_POS])
		{
		case 'A' :
			/* read length directly */
			sfdo->len = atol( &(sfdo->header[SFDU_LEN_POS]) );
			break;
		case 'S' :
			/* This function searches ahead in the file for the
			** marker label containing the given marker
			** string.  On return, the fp is set at the
			** beginning of the SFDU label containing the
			** marker.  The function returns the fseek value
			** at that position.
			*/
			sfdo->len = (ulong)(find_end_marker(sfdo_filep(sfdo),
					&(sfdo->header[SFDU_LEN_POS]) )
					- start_pos);
/* ~
			DEBUG_PRN("DEBUG: Object length = %lu\n", sfdo->len);
*/
			break;
		case 'F' :
			/* save file position, then seek ahead to EOF.
			** set length to EOF posn - start_pos.
			*/
			fseek(sfdo_filep(sfdo), 0, SEEK_END);
			sfdo->len = ftell(sfdo_filep(sfdo)) - start_pos;
			fseek(sfdo_filep(sfdo), start_pos, SEEK_SET);
			break;
		default:
			break;
		}
	}
	sfdo->unread = sfdo->len;
	sfdo->sfd->unread -= SFDU_HEADER_LEN; /* ??? */

	/* check class and setup for reads depending on what it is */
    /* see defines in sfdu.h */
	switch (sfdo->class)
	{
	case SFDUCID_CATALOG:
		/* K - expect catalog info, let caller read it */
	case SFDUCID_INFO:
		/* I - value field contains data in sub SFDOs */
	case SFDUCID_SUPPL:
	case SFDUCID_TEXT:
		/* T - value field contains readable text */
	case SFDUCID_DDP:
		/* D - value field contains data descrip. package */
	case SFDUCID_DEDICT:
	case SFDUCID_DFDESC:
	case SFDUCID_DLANG:
	case SFDUCID_OBJECT:
	case SFDUCID_VOLUME:
		break;
	case SFDUCID_UNCLAGR:
		break;
	case SFDUCID_AGR:
		/* Z - read data part to determine what kind of aggregation */
		break;
	default:
		return(SFDU_ERROR);
	}
	return((SFDU_CLASS)sfdo->class);
}
//------------------------------------------------------------------------
SFDU_CLASS
sfdu_next_sub_sfdo(parent_sfdo, sfdo)
SFDO * parent_sfdo;
SFDO * sfdo;
{
	char buff[SFDU_HEADER_LEN+1];

	/* initialize the sfdo by copying the parent_sfdo pointer into it */
	sfdo->sfd = parent_sfdo->sfd;
	/* read and validate header */
	if ((int)fread(buff, 1, SFDU_HEADER_LEN, sfdo->sfd->fd) < SFDU_HEADER_LEN)
	{
    		/* error -- unexpected EOF */
		fprintf(stderr, "Unexpected EOF: incomplete SFDU\n");
    		return(SFDU_ERROR);
	}
	strncpy(sfdo->header, buff, SFDU_HEADER_LEN);
	sfdo->header[SFDU_HEADER_LEN] = CHARNULL;
	sfdo->class = buff[SFDU_CLASS_POS];
	sfdo->version = buff[SFDU_VERSION_POS];

   	sfdo->len = atol(&buff[SFDU_LEN_POS]);

	strncpy(sfdo->ddpid, &buff[SFDU_DDPID_POS], SFDU_DDPID_LEN);
	sfdo->ddpid[SFDU_DDPID_LEN] = CHARNULL;
	sfdo->unread = sfdo->len;
	sfdo->sfd->unread -= SFDU_HEADER_LEN;

	/* check class and setup for reads depending on what it is */
	switch (sfdo->class)
	{
	case SFDUCID_CATALOG:
		/* K - expect catalog info, let caller read it */
	case SFDUCID_INFO:
		/* I - value field contains data in sub SFDOs */
	case SFDUCID_SUPPL:
	case SFDUCID_TEXT:
		/* T - value field contains readable text */
	case SFDUCID_DDP:
		/* D - value field contains data descrip. package */
	case SFDUCID_DEDICT:
	case SFDUCID_DFDESC:
	case SFDUCID_DLANG:
	case SFDUCID_OBJECT:
	case SFDUCID_VOLUME:
		break;
	case SFDUCID_UNCLAGR:
		break;
	case SFDUCID_AGR:
		/* Z - read data part to determine what kind of aggregation */
		break;
	default:
		return(SFDU_ERROR);
	}

	return(sfdo->class);
}

//--------------------------------------------------------------------------
long
sfdu_sfdo_read(buffer, size, count, sfdo)
void * buffer;
long size;
long count;
SFDO * sfdo;
{
	int n,bytes;

	if (sfdo->unread <= 0)
		return((ulong)0);

	bytes = size * count;
	bytes = (bytes > sfdo->unread ? sfdo->unread : bytes);
	n = fread(buffer, 1, bytes, sfdo_filep(sfdo));
	if (n > 0)
	{
		sfdo->unread -= n;
		sfdo->sfd->unread -= n;
		return(n/size);
	} else
		return(n);
}
//---------------------------------------------------------------
char *
sfdu_fgets(buffer, length, sfdo)
char * 	buffer;
register int	length;
SFDO * sfdo;
{
	int n;

	if (sfdo->unread <= 0)
		return(NULL);

	length = (length > sfdo->unread ? sfdo->unread : length);
//    printf ("sfdu_fgets: length = %d\n",length);
	if (*fgets(buffer, length, sfdo_filep(sfdo)) != '\0')
	{
		n = strlen(buffer);
		sfdo->unread -= n;
		sfdo->sfd->unread -= n;
	}

	return(buffer);
}
//----------------------------------------------------------------------
int
sfdu_close(sfdu)
SFDU * sfdu;
{
	fclose(sfdu->fd);
	free(sfdu);
	return(0);
}


/*
** TEST PROGRAM
** This module if compiled using -DUNIT_TEST will compile into a
** simple test program that will open a file named on the command
** line and try to read it as an SFDU. It will report statistics
** and information about the SFDOs it opens, and will print out
** all the data.
*/
#ifdef UNIT_TEST

char	progname[80];
char	filename[80];
SFDU *  sfdu;
SFDO	sfdo;
char	buff[1000];

main(argc, argv)
int	argc;
char ** argv;
{
	strcpy(progname, argv[0]);
	strcpy(filename, argv[1]);
	fprintf(stderr, "%s:working on %s\n", progname, filename);

/* open up the file given in arg1 as an SFDU */
	if ((sfdu = sfdu_open(filename, "r")) == NULL)
	{
		fprintf(stderr, "%s:failed to open file %s\n",
			progname, filename);
		exit(1);
	}

/* print header and length */
/* print its.class (should be 'Z') */
	fprintf(stderr, "\nsfdu header = %s\n", sfdu_label(sfdu));
	fprintf(stderr, "sfdu size = %lu\n", sfdu_len(sfdu));
	fprintf(stderr, "sfdu class = %c\n", sfdu_class(sfdu));
	fprintf(stderr, "sfdu version = %c\n", sfdu_version(sfdu));
	fprintf(stderr, "sfdu ddpid = %s\n", sfdu_ddpid(sfdu));

	fprintf(stderr, "POS = %ld\n", (long)ftell(sfdu->fd));

	while ( sfdu_next_sfdo(sfdu, &sfdo) != SFDU_ERROR )
	{
		ulong n,i;

		/* print header.class and length */
		fprintf(stderr, "\n\nsfdo header = %s\n", sfdu_label(&sfdo));
		fprintf(stderr, "sfdo size = %lu\n", sfdu_len(&sfdo));
		fprintf(stderr, "sfdo class = %c\n", sfdu_class(&sfdo));
		fprintf(stderr, "sfdo version = %c\n", sfdu_version(&sfdo));
		fprintf(stderr, "sfdo ddpid = %s\n", sfdu_ddpid(&sfdo));

#ifdef TEST0A
		/* get contents */
		n = sfdu_sfdo_read(buff, (long)1, (long)999, &sfdo);

	/* print contents */
		switch (sfdu_class(&sfdo)) {
		case SFDUCID_CATALOG :
		case 'I' :
		/* if.class = 'K' just print data */
			for (i=0; i<n; i++) 
				fprintf(stderr, "%c", buff[i]);
			fprintf(stderr, "|END|\n\n");
			break;
		case 'R' :
		/* if.class = 'R' print data and everything else... */
			for (i=0; i<n; i++) printf("%c", buff[i]);
			fprintf(stderr, "\n END OF SFDO -- DATA BLOCK BEGINS\n");
			while (sfdu_fgets(buff, 100, &sfdo))
				fprintf(stderr, "%s", buff);
			break;
		default  :
			fprintf(stderr, "Unknown class: %c\n", sfdu_class(&sfdo));
			break;
		}
#else
		while ( *sfdu_fgets(buff, 90, &sfdo) != '\0' )
			fprintf(stderr, "-->%s\n", buff);
		fprintf(stderr, "|END|\n\n");
		fprintf(stderr, "sfdo.unread = %ld\n", sfdo.unread);
		/* seek to end of sfdo */
		sfdo_end(&sfdo);
#endif
	}

	sfdu_close(sfdu);

	DEBUG_PRS("DEBUG: All Done!");
	fflush(stderr);
	exit(0);
}
#endif

