/***********************************************************************
    TITLE:	read_sclk.c (read SFDU-formatted sclkscet file)
CIN:	read_sclk.c,v 1.1.1.1 1994/08/31 19:22:02 root Exp

FILE NAME:	read_sclk.c
AUTHOR:		David A. Wagner
CREATED: 	January 4, 1991
LAST UPDATE: 	1994/08/31 19:22:02
LAST UPDATE:    11-18-2021 Ray Bambery
            removed some debug statements added 6-7-2021 see //// by Ray Bambery
                12-07-2021 - Ray Bambery
LAST UPDATE:    Dec 07, 2021 - Ray Bambery
VERSION:    1.9
COMPILER:   gcc 7.2 on Centos 7 (pistol.jpl.nasa.gov)
OS:         Solaris 10.2 and Centos 7 

REFERENCES:
WARNINGS:	Currently assumes all SCLKSCET files are in SFOC SFDU format
BUGS:
UNIT TEST:	Compile with -DDEBUG and -DUNIT_TEST
WAIVERS:

DESCRIPTION:
** Open named SCLKSCET file and read the data elements into an
** internal table having the following structure:
**
**	typedef struct sclktable_t {
**    		double	scet0;
**		double 	sclk0;
**		double	dbsclk;	     as defined in sclkscet.h
**		double	sclkrate;
**		double	dut;
**	} sclktbl, *sclktblptr;   
**
** The user may then use the associated routines sclktoscet or
** scettosclk to convert time values using this table.  
** This routine will allocate memory for the table (using malloc)
** by estimating how many table rows will be needed by scanning
** the file.
**
** The return value is the number of table entries if everything worked, 
** or -1 otherwise.

CHANGE HISTORY:
  read_sclk.c,v

 * Revision 1.8 Centos 7/Solaris 10.2 - Ray Bambery
 * add include <stdlib.h> for atof  and atoi - msg:  
 * warning: implicit declaration of function ‘atof’
 * Change BUFSIZ to BUFSIZE to avoid conflict with
 * /usr/include/malloc.h:25:0,
 *               from read_sclk.c:86:
 * /usr/include/stdio.h:127:0: note: this is the location of the previous definition
 * #define BUFSIZ _IO_BUFSIZ
 * 
 * Revision 1.1.1.1  1994/08/31  19:22:02  root
 * Initial Release
 *
 * Revision 1.8  1992/12/31  04:17:12  kevin
 * corrected RCS keywords
 *
 * Revision 1.7  1991/10/07  21:37:39  dsfa
 * corrected strcpy(strtok()) error.
 *
 * Revision 1.6  1991/08/05  21:32:19  dsfa
 * added check for SPACECRAFT_NUM and SPACECRAFT_ID
 *
 * Revision 1.5  1991/07/03  19:54:24  dsfa
 * *** empty log message ***
 *
 * Revision 1.4  1991/04/18  16:21:03  dsfa
 * added DEBUG statements
 *
 * Revision 1.3  1991/04/11  00:05:57  dsfa
 * changed maxrows = length/70 to length/60
 *
 * Revision 1.2  1991/03/28  22:36:13  dsfa
 * correct debug MSG warnings
 *
 * Revision 1.1  1991/03/13  17:41:19  dsfa
 * Initial revision
 *
***********************************************************************/
//      routine                 called_by                   located
//      read_sclk               sclk2scet, vgr_scet, load_file
//      DEBUG_PRN               read_sclk                   scegen.h,sfdu.h    
//      DEBUG_PRN2              read_sclk                   scegen.h,sfdu.h
//      DEBUG_PRS               read_sclk                   scegen.h,sdfu.h
//
//
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>   /* for atof and atoi */
#include "sclkscet.h"
#include "scegen.h"
/* string containing parsing delimiters for strtok */
#define DELIMITERS	" \n\r"
#define KVNDELIM	";= \n\r"

sclktblptr	SCLK_TABLE = (sclktblptr)NULL;
int		sclk_rows = 0;
int		sclk_mission = 0;

struct	keyword_struct  keyword[MAX_NUM_KEYWORDS];

int	keyword_rows = 0;

static char	buffer[BUFSIZE];
extern int any_time_to_seconds(register char* s, double * t);

//--------------------------------------------------------------------
int
read_sclk(sfdu, m_id, s_id)
	SFDU	* sfdu;   /* SCLK_SCET.XX */ 
	int	m_id; 
	int	*s_id; /* pointer to spacecraft_id */ 
{
/*   already opened SCLK_SCET (*sfdu), read and parse the lines (sfdu type 3 file)
 *   1st record is CCSD3ZS00001$$sclk$$NJPL3KS0L015$$scet$$
 *   Validate label and  mission and parse table, store final entries in SCLK_TABLE
 *   pointed to by sclktblptr or scklkentry 
 */
	SFDO		sfdo;
	char		* c;
	int		maxrows;
	int		i = 0;
	sclktblptr	sclkentry;	/* current entry */
	double		this_sclk, last_dbsclk, dboffset;
	sclktblptr	xtra_memory;
    double sclkrate;
/*
 * First open the SCLKSCET SFDU file using the sfdu library 
 * function.
 */
	/* Read the K-class SFDO and parse the catalog info */
	if (sfdu_next_sfdo(sfdu, &sfdo) == SFDU_ERROR)
	{
		/* error -- cannot find K-object */
		DEBUG_PRS("DEBUG: Failed to open first SFDO\n");
		return(ERROR);
	}
/*
 * Parse the K-object and validate the values.  Also, find the
 * MISSION_NAME and set the value of m_id for the function
 * any_sclk_to_double().
 */
	if (sfdu_class(&sfdo) != 'K')
	{
		DEBUG_PRS("DEBUG: Expected K object first\n");
		return(ERROR);
	}
	if (sfdu_sfdo_read(buffer, (long)1, (long)BUFSIZE, &sfdo) <= 0)
	{
		DEBUG_PRS("DEBUG: Failed to read K object data\n");
		return(ERROR);
	}
	if ((c = strstr(buffer, "MISSION_NAME")) == NULL)
	{
		DEBUG_PRS("DEBUG: Failed to find MISSION_ID\n");
		return(ERROR);
	}

	/* skip forward to the '=' sign, then skip any spaces */
	c = strtok(c, KVNDELIM);
	strcpy(keyword[i].name, c);
	c = strtok(NULL, KVNDELIM);
	strcpy(keyword[i].value, c); 
	
	/* match MISSION_NAME values */
	if      (strncmp(keyword[i].value, "VOYAGER", 7) == 0)
		sclk_mission = VOYAGER;
	else if (strncmp(keyword[i].value, "GALILEO", 7) == 0)
		sclk_mission = GALILEO;
	else if (strncmp(keyword[i].value, "ULYSSES", 7) == 0)
		sclk_mission = ULYSSES;
	else if (strncmp(keyword[i].value, "MAGELLAN", 8) == 0)
		sclk_mission = MAGELLAN;
	else if (strncmp(keyword[i].value, "MARS", 4) == 0)
		sclk_mission = MARS_OBSERVER;

	if (sclk_mission == 0)
	{
		DEBUG_PRS("DEBUG: No match on MISSION_NAME value\n");
		return(ERROR);
	}

/*  We compare the mission_id taken from the SCLKvSCET file
 *  against the mission_id value inputted by the user (via the command
 *  line).  If they do not match, an error message is displayed, this
 *  function returns ERROR (-1) to the calling program.
 */
	if (sclk_mission != m_id)
	{
		fprintf(stderr,"READSCLK Error - MISSION INVALID in SCLKvSCET FILE\n");
		fprintf(stderr,"Mission ID=%d inputted by user not the same\
            as Mission ID=%d obtained from SCLKvSCET file\n", m_id,
		   sclk_mission); 
		return(ERROR);
	}
/*
 * if mission name and spacecraft id are found then continue to read the rest of the
 * k-class header values and store them in global array keyword[].
 */
	while(keyword[i].name != NULL)
	{
		i++;
		c = strtok(NULL, KVNDELIM);
		if (c == NULL) break;
		strcpy(keyword[i].name, c);
		c = strtok(NULL, KVNDELIM);
		if (c == NULL) break;
		strcpy(keyword[i].value, c);
		if (strcmp(keyword[i].name,"DSN_SPACECRAFT_NUM") == 0)
			*s_id = atoi(keyword[i].value);
		if (strcmp(keyword[i].name,"SPACECRAFT_ID") == 0)
			*s_id = atoi(keyword[i].value);
	}
	keyword_rows = i;

/*
 * open the I-object
 */
	if (sfdu_next_sfdo(sfdu, &sfdo) == ERROR)
	{
		/* error -- cannot find I-object */
		DEBUG_PRS("DEBUG: Failed to open second SFDO\n");
		return(ERROR);
	}
/*
 * malloc space for the table using the sfdo length
 * field as an estimate of how many rows to expect:
 * sclk_rows = length/50
 */
	maxrows = (int)(sfdu_len(&sfdo)/50L)+20;
//    printf ("read_sclk:       maxrows = %d\n",maxrows);  //removed RJB - 5-17-2021
/*    printf ("-----------calloc  -----------------");  
*/
	if ((SCLK_TABLE =
		(sclktblptr)calloc(maxrows, sizeof(sclktbl))) == NULL)
	{
		/* error -- no table allocated */
		DEBUG_PRS("DEBUG: Failed to calloc table space\n");
		return(ERROR);
	}
////    printf ("sizeof (sclktbl) = %d\n",sizeof(sclktbl));
/*
 * Parse the data entries into our table.
 * *____SCLK0_______  ____SCET0____________ _DUT__ _SCLKRATE___ 
 */
	dboffset = last_dbsclk = this_sclk = 0.0;

//// printf("SCLK_TABLE in read_sclk = %d \n",SCLK_TABLE);       //removed RJB - 5-17-2021

	sclkentry = SCLK_TABLE;
//    printf ("read_sclk: maxrows = %d\n",maxrows);
	for ( sclk_rows = 0; sclk_rows < maxrows; )  /* maxrows =  */
	{
//        printf ("read_sclk: START  sclk_rows = %d\n",sclk_rows);       //removed RJB - 5-17-2021
//        fflush;
//        check for EOF
		if (*(c=sfdu_fgets(buffer, 90, &sfdo)) == CHARNULL)
			break;
////    printf ("read_sclk: before continue  buffer = %s\n",buffer);

		if (*c == '*')	continue;	/* comment line */

		while(isspace(*c)) c++;		/* skip leading spaces */
		if (*c == CHARNULL)	continue;	/* blank line */

		c = strtok(c, DELIMITERS);

		/* read the SCLK value */
		if (any_sclk_to_double(sclk_mission, c, &this_sclk) 
			== ERROR)
		{
			DEBUG_PRN("DEBUG Failed to parse SCLK value at %s\n", c);
			return(ERROR);
		}

////printf("____SCLK0___ %s    this_sclk= %lf \n", c, this_sclk);       //removed RJB - 5-17-2021
 
		sclkentry->sclk0 = this_sclk;
		/* calculate the dbsclk value */
		if (this_sclk + dboffset < last_dbsclk)
		{
			/* calculate a new offset */
			dboffset = last_dbsclk;
		}
		sclkentry->dbsclk0 = last_dbsclk = this_sclk + dboffset;

		c = strtok(NULL, DELIMITERS);

		/* read the SCET value */
		if (any_time_to_seconds(c, &(sclkentry->scet0)) == ERROR) 

/*		if (std_time_to_seconds(c, &(sclkentry->scet0)) == ERROR) */
		{
			DEBUG_PRN("DEBUG: Failed to parse TIME value at: %s\n", c);
			return(ERROR);
		}

////printf("____SCET0____ %s    sclkentry->scet0 = %lf \n",         //removed RJB - 5-17-2021
////        c, sclkentry->scet0);

		c = strtok(NULL, DELIMITERS);

		/* read the DUT value */
		sclkentry->dut = atof(c);
/*
printf ("_DUT__ %s  sclkentry->dut = %lf \n",c,sclkentry->dut);     //removed RJB - 5-17-2021
*/
		c = strtok(NULL, DELIMITERS);

		/* read the SCLKRATE value */
		sclkentry->sclkrate = atof(c);
        sclkrate = sclkentry->sclkrate;
////  printf ("_SCLKRATE___ %s sclkentry->sclkrate %lf\n",c,sclkentry->sclkrate);     //removed RJB - 5-17-2021
////  printf ("  sclkentry = %d\n",sclkentry);
		sclkentry++;
		sclk_rows++;
//        if (sclk_rows >= 20) {break;}
        fflush;
////        printf ("  sclkentry = %d sclk_erows = %d\n",sclkentry, sclk_rows);         //removed RJB - 5-17-2021

	}
////    printf ("AFTER LOOP\n");
// /*
    sclkentry--;                    // added 5-14-2021 - RJB
/*   //removed RJB - 5-17-2021
	printf ("DEBUG: last_sclk0    = %lf\n",  sclkentry->sclk0);
	printf ("       last_dbsclk0  = %lf\n",  sclkentry->dbsclk0);
	printf ("DEBUG: last_scet0    = %lf\n",  sclkentry->scet0);
	printf ("       last_dut      = %lf\n",  sclkentry->dut);
	printf ("DEBUG: last_sclkrate = %lf  %lf\n",  sclkentry->sclkrate,sclkrate);
	printf ("DEBUG: %d rows\n", sclkrows-1);
	printf ("DEBUG: number of rows in sclkscet file is %d\n", sclk_rows);
*/
	sfdu_close(sfdu);
	return(sclk_rows);
}
