/***********************************************************************
TITLE:	anysclk.c: spacecraft clock conversion functions
CIN:	anysclk.c,v 1.1.1.1 1994/08/31 19:22:02 root Exp

FILE NAME:	/u3/cvs_release/src/VGR_EDR/scet/anysclk.c,v
AUTHOR:		David A. Wagner
CREATED: 	January 4, 1991
LAST UPDATE: 	Dec 07, 2021 - Ray Bambery
VERSION:	1.6
COMPILER:	gcc 7.2 on Centos 7 (pistol.jpl.nasa.gov)
OS:		    Solaris 10.2 and Centos 7 
REFERENCES:	SYS-Any-TimeForms for ASCII formats
WARNINGS:	Curently knows VGR, GLL, ULS, MGN, MO formats
BUGS:
UNIT TEST:	Compile with -DDEBUG and -DUNIT_TEST
WAIVERS:

CHANGE HISTORY:
  anysclk.c
 * Revision 1.6 Centos 7/Solaris 10.2 - Ray Bambery
 * add include <stdlib.h> for atof - msg:  
 * warning: implicit declaration of function ‘atof’
 *
 * Revision 1.1.1.1  1994/08/31  19:22:02  root
 * Initial Release
 *
 * Revision 1.5  1993/11/10  17:47:16  dsfa
 * Fixed rounding bugs for GLL,VGR,ULS; fixed VGR linecount modulus.
 *
 * Revision 1.4  1992/08/18  16:18:27  dsfa
 * Improved rounding in any_sclk_from_double.
 *
 * Revision 1.3  1991/10/04  00:02:39  dsfa
 * Rounded up VOYAGERs mod800 count.
 *
 * Revision 1.2  1991/07/03  19:48:14  dsfa
 * *** empty log message ***
 *
 * Revision 1.1  1991/03/13  17:38:02  dsfa
 * Initial revision
 *
***********************************************************************/
/*
** MANUAL
**	any_sclk_to_double 3 "1.1.1.1"
** NAME
**	any_sclk_to_double() -- converts SCLK string to double
** SYNOPSIS
**	int any_sclk_to_double(mission, string, sclk)
**		int mission;
**		char * string;
**		double * sclk;
**
**	int any_sclk_from_double(mission, sclk, string)
**		int mission;
**		double sclk;
**		char * string;
**
** DESCRIPTION
**	These functions are intended to allow programs to parse strings
**	containing spacecraft clock values in SFOC standard format
**	into a single number and back.  The numerical value is a double
**	containing the calculated number of major counts for the given
**	spacecraft clock.  The clock format is identified by the __ mission
**	argument and may have any of the following values:
**
**	 
** 	 0	VOYAGER		mod16.mod60.mod800
**	 1	GALILEO		rim.mod91.mod10.mod8
**	 2	VOYAGER 	(also)
**	 3	ULYSSES		major.frac
**	 4	MAGELLAN	rim.mod91.mod10.mod8
**	 5	MO		seconds.mod256
**
**	any other value will result in an error.
**
** AUTHOR
**	David A. Wagner, Jet Propulsion Laboratory
**
** SEE ALSO
**	sclktoscet() scettosclk()
*/
//      routine                 called_by
//      any_sclk_to_double      read_sclk
//      any_sclk_from_double
//
//
#include <stdio.h>
#include <stdlib.h>    /* for atof */
#include <math.h>

#ifndef ERROR
#define ERROR	(-1)
#endif
//----------------------------------------------------------------------------
int
any_sclk_to_double(mission, input, output)
	int		mission;
	register char	* input;
	double		* output;
{
	long		field1, field2, field3, field4;

	/* ASSUME: input is pointing at first character of SCLK string
	** which should be a digit 
	*/
	field1 = field2 = field3 = field4 = 0L;

	switch(mission)
	{
	case 5 :	/* MARS_OBSERVER */
		if (sscanf(input, "%ld.%ld", &field1, &field2) < 1)
			return(ERROR);
		*output = (double)field1 + ((double)field2)/256.0;
		break;
	case 0 :	/* VOYAGER */
	case 2 :
		if (sscanf(input, "%ld.%ld.%ld",
			&field1, &field2, &field3) < 1)
			return(ERROR);
		*output = (double)field1 + (((double)field2 + 
			(double)(field3-1)/800.0)/60.0);
		break;
	case 3 :	/* ULYSSES */
		*output = atof(input);
		break;
	case 4 :	/* MAGELLAN */
	case 1 :	/* GALILEO - same as MGN */
		if (sscanf(input, "%ld.%ld.%ld.%ld",
			&field1, &field2, &field3, &field4) < 1)
			return(ERROR);
		*output = (double)field1
				+ ( ( (double)field4 /8.0
				+ (double)field3 ) /10.0
				+ (double)field2 ) /91.0;
		break;
	default:
		return(ERROR);
	}
	return(1);
}

//----------------------------------------------------------------
int
any_sclk_from_double(mission, input, output)
	int		mission;
	double		input;
	char		* output;
{
	long		field1, field2, field3, field4;

	switch(mission)
	{
	case 5 :	/* MARS_OBSERVER */
		field1 = floor(input);		/* seconds count */
		input -= (double)field1;
		input *= 256.0;
/*		field2 = nint(input);		*//* mod256 */
		field2 = (int)(input);		/* mod256 */
		if(field2 > 255)	/* account for round up case */
		{
			field2 -= 256;
			field1++;
		}
		sprintf(output, "%010ld.%03ld", field1, field2);
		break;
	case 0 :	/* VOYAGER */
	case 2 :
		field1 = floor(input);		/* mod16 count */
		input -= (double)field1;
		input *= 60.0;
		field2 = floor(input);		/* mod60 count */
		input -= (double)field2;
		input *= 800.0;
		input += .1;
/*		field3 = nint(input) + 1; */		/* RTI */
		field3 = (int)(input) + 1;		/* RTI */
		if (field3 > 800) {
		    field3 = 0;
		    if (++field2 > 60) {
			field2 = 0;
			field1++;
		    }
	 	}
		sprintf(output, "%05ld.%02ld.%03ld", field1, field2, field3);
		break;
	case 3 :	/* ULYSSES */
		field1 = floor(input);		/* major count */
		input -= (double)field1;
		input *= 10.0;
/*		field2 = nint(input);		*//* fraction */
		field2 = (int)(input);		/* fraction */

		sprintf(output, "%010ld.%1ld", field1, (field2 > 4 ? 5 : 0));
		break;
	case 4 :	/* MAGELLAN */
	case 1 :	/* GALILEO - same as MGN */
		field1 = floor(input);		/* RIM */
		input -= (double)field1;
		input *= 91.0;
		field2 = floor(input);		/* mod91 */
		input -= (double)field2;
		input *= 10.0;
		field3 = floor(input);		/* mod10 */
		input -= (double)field3;
		input *= 8.0;
/*		field4 = nint(input);		*//* mod8 */
		field4 = (int)(input);		/* mod8 */
		/* propagate rounding */
		if (field4 > 7) {
		    field4 = 0;
		    if (++field3 > 9) {
			field3 = 0;
			if (++field2 > 90) {
			    field2 = 0;
			    ++field1;
			}
		    }
		}
		sprintf(output, "%09ld.%02ld.%1ld.%1ld",
			field1, field2, field3, field4);
		break;
	default:
		return(ERROR);
	}
	return(0);
}



