/******************************************************************************
                          timelib.cpp  -  description
                             -------------------
    begin                : Tue Jul 24 13:12:00 CEST 2001

    author               : Hans Vaith
    email                : hav@mpe.mpg.de
    copyright            : (C) 2001 by Max-Planck-Institut fr extra-
                           terrestrische Physik, D-85740 Garching

    Cluster-II EDI: functions for conversion of time formats
******************************************************************************
$Id: timelib.cpp,v 1.2 2005/04/07 08:34:43 hav Exp $
$Log: timelib.cpp,v $
Revision 1.2  2005/04/07 08:34:43  hav
more dev

Revision 1.1.1.1  2002/10/14 17:08:19  hav
c/c++ sources

Revision 1.5  2001/10/09 16:21:40  hav
made conversion functions to work for year 2000 also

Revision 1.4  2001/08/08 19:44:02  hav
added checks in TL_Set()

Revision 1.3  2001/08/02 11:59:19  hav
ls -l *.cpp
previous line is nonsense
change: single line output of TL_PrintDateTime()

Revision 1.2  2001/08/01 15:15:21  hav
extended library

Revision 1.1  2001/07/25 14:58:30  hav
Initial revision

******************************************************************************/

static char rcsid[] = "$Id: timelib.cpp,v 1.2 2005/04/07 08:34:43 hav Exp $";

/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <stdlib.h>
#include "timelib.h"

/******************** CONSTANTS AND MACROS ***********************************/

/******************** TYPE DEFINITIONS ***************************************/

/******************** FUNCTION PROTOTYPES ************************************/

static void TL_Normalize(scet *tp);
static void TL_Ct2DateTime(scet *tp);
static void TL_DateTime2Ct(scet *tp);

/******************** FUNCTION DEFINITIONS ***********************************/

// ============================================================================
   void TL_Set(scet *tp, short year, short month, short day,
                         short hr, short min, short sec,
                         short msec, short usec, short nsec)
// ============================================================================
{
   const char *fname = "TL_Set()";

   tp->year = year;
   tp->month = month;
   tp->day = day;
   tp->hr = hr;
   tp->min = min;
   tp->sec = sec;
   tp->msec = msec;
   tp->usec = usec;
   tp->nsec = nsec;

   if (month < 1 || day < 1 ||
       hr < 0 || min < 0 || sec < 0 ||
       msec < 0 || usec < 0 || nsec < 0 ) {
      tp->ct = 0.0;
      fprintf(stderr, "%s: bad time specification\n", fname);
      TL_PrintDateTime(tp, stderr);
      exit(1);
   }


   // calculate doy (day of year) from day and month
   short montharr[] = { 31, 28, 31, 30,
                        31, 30, 31, 31,
                        30, 31, 30, 31 };
   if (tp->year%4 == 0) montharr[1] = 29;

   tp->doy = 0;
   short month_cnt = 1;
   while (month_cnt < tp->month) {
      tp->doy += montharr[month_cnt-1];
      ++month_cnt;
      if (month_cnt > 12) {
         fprintf(stderr, "%s: calculated bad value for month: %hd\n",
                        fname, month_cnt);
         TL_PrintDateTime(tp, stderr);
         exit(1);
      }
   }
   tp->doy += tp->day;

   // calculate ct
   TL_DateTime2Ct(tp);

}


// ============================================================================
   void TL_SetCt(scet *tp, long double value)
// ============================================================================
{
   tp->ct = value;
   TL_Ct2DateTime(tp);
}

// ============================================================================
   void TL_Add(scet *tp, int what, int value)
// ============================================================================
{
   switch (what) {
   case TL_DAY   : tp->doy += value; break;
   case TL_HOUR  : tp->hr += value; break;
   case TL_MIN   : tp->min += value; break;
   case TL_SEC   : tp->sec += value; break;
   case TL_MSEC  : tp->msec += value; break;
   case TL_USEC  : tp->usec += value; break;
   case TL_NSEC  : tp->nsec += value; break;
   default :
      fprintf(stderr, "TL_Add(): bad value for 'what' : %d\n", what);
      exit(1);
   }

   TL_Normalize(tp);
   TL_DateTime2Ct(tp);

}


// ============================================================================
   void TL_AddCt(scet *tp, long double value)
// ============================================================================
{
   tp->ct += value;
   TL_Ct2DateTime(tp);
}



// ===========================================================================
   void TL_PrintDateTime(scet *tp, FILE *fp)
// ===========================================================================
{
   if (0) printf("%s\n", rcsid); // dummy instruction

   if (fp==NULL) {
      fprintf(stderr, "TL_PrintDateTime(): fp is NULL! No output generated.\n");
      return;
   }

   fprintf(fp, "%04hd-%02hd-%02hd %02hd:%02hd:%02hd "
               "%03hd.%03hd.%03hd (doy=%03hd) "
               "(ct=%20.9Lf)\n",
               tp->year, tp->month, tp->day,
               tp->hr, tp->min, tp->sec,
               tp->msec, tp->usec, tp->nsec,
               tp->doy, tp->ct);

}

// ===========================================================================
   void TL_PrintDate(scet *tp, FILE *fp)
// ===========================================================================
{
   if (0) printf("%s\n", rcsid); // dummy instruction

   if (fp==NULL) {
      fprintf(stderr, "TL_PrintDate(): fp is NULL! No output generated.\n");
      return;
   }

   fprintf(fp, "%04hd-%02hd-%02hd\n",
               tp->year, tp->month, tp->day);

}

// ============================================================================
   int TL_DateCmp(scet t1, scet t2)
// ============================================================================
// compares the dates (year/month/day) of t1 and t2
// Return value:
//         -1 if t1 is earlier than t2
//          0 if the dates are equal
//          1 if t1 is later than t2
// ****************************************************************************
{
   if (t1.year < t2.year) return -1;
   else if (t1.year > t2.year) return 1;

   // years are equal
   if (t1.month < t2.month) return -1;
   else if (t1.month > t2.month) return 1;

   // year and month are equal
   if (t1.day < t2.day) return -1;
   else if (t1.day > t2.day) return 1;

   return 0; // dates are equal
}


// ===========================================================================
   static void TL_Normalize(scet *tp)
// ===========================================================================
{
   const char *fname = "TL_Normalize()";
   int carry;

   carry = tp->nsec/1000;
   tp->nsec %= 1000;

   tp->usec += carry;
   carry = tp->usec/1000;
   tp->usec %= 1000;

   tp->msec += carry;
   carry = tp->msec/1000;
   tp->msec %= 1000;

   tp->sec += carry;
   carry = tp->sec/60;
   tp->sec %= 60;

   tp->min += carry;
   carry = tp->min/60;
   tp->min %= 60;

   tp->hr += carry;
   carry = tp->hr/24;
   tp->hr %= 24;

   tp->doy += carry;

   int daymax;
   while (1) {
      if (tp->year %4 == 0) daymax = 366; else daymax = 365;
      if (tp->doy <= daymax) break;
      ++tp->year;
      tp->doy -= daymax;
   }


   // calculate month and day of month
   short montharr[] = { 31, 28, 31, 30,
                        31, 30, 31, 31,
                        30, 31, 30, 31 };
   if (tp->year%4 == 0) montharr[1] = 29;

   tp->day = tp->doy;
   tp->month = 0;
   while (tp->day > montharr[tp->month]) {
      tp->day -= montharr[tp->month];
      ++tp->month;
      if (tp->month > 11) {
         fprintf(stderr, "%s: calculated bad value for month: %hd\n",
                        fname, tp->month);
         TL_PrintDateTime(tp, stderr);
         exit(1);
      }
   }
   ++tp->month ; // Jan - Dec is 1 - 12

}

// ===========================================================================
   static void TL_Ct2DateTime(scet *tp)
// ===========================================================================
// convert from ctime (seconds since the stroke of midnight on 1-Jan-1970)
// to year/month/day/hr/min/sec...
// ***************************************************************************
{
   const char *fname = "TL_Ct2Datetime()";

   const long sec_per_day = 86400;
//   long double ct_2001 = tp->ct -  978307200.0; // seconds since 1-Jan-2001
   long double ct_2000 = tp->ct -  946684800.0; // seconds since 1-Jan-2000

   if (ct_2000 < 0) {
      fprintf(stderr, "%s: too lazy to handle times prior to 1-Jan-2000\n",
                       fname);
      exit(1);
   }

   tp->year = 2000;
   tp->doy  = (short)(ct_2000/sec_per_day); // zero-based ! corrected further down.
   tp->hr   = (short)((ct_2000 - sec_per_day*tp->doy) / 3600);
   tp->min  = (short)((ct_2000 - sec_per_day*tp->doy - 3600*tp->hr) / 60);
   tp->sec  = (short)(ct_2000 - sec_per_day*tp->doy - 3600*tp->hr - 60*tp->min);
   long double subsec = ct_2000 - (long)ct_2000;
   tp->msec = (short)(1000*subsec);
   tp->usec = (short)(1000* (1000*subsec - tp->msec) );
   tp->nsec = (short)(1000* (1000000*subsec - 1000*tp->msec - tp->usec) );

   ++tp->doy;
   TL_Normalize(tp); // calculates also day and month

}


// ===========================================================================
   static void TL_DateTime2Ct(scet *tp)
// ===========================================================================
// convert from year/month/day/hr/min/sec...
// to ctime (seconds since the stroke of midnight on 1-Jan-1970)
// ***************************************************************************
{
   long days = tp->doy - 1;     // full days in current year
   short year = tp->year;

   if (tp->year < 2000) {
     fprintf(stderr, "TL_DateTime2Ct(): dates before 1-Jan-2000 are "
      "not accepted!\n");
     exit(1);
   }

   while (year > 2000) {
      --year;
      if ((year % 4) == 0)
         days += 366;
      else
         days += 365;
   }

   //   tp->ct = 978307200.0 +       // seconds from 1-Jan-1970 to 1-Jan-2001
   tp->ct = 946684800.0 +       // seconds from 1-Jan-1970 to 1-Jan-2000
      ((long double) days * 24 + tp->hr) * 3600 + tp->min * 60 + tp->sec +
      1.0e-3 * tp->msec + 1.0e-6 * tp->usec + (long double)1.0e-9 * tp->nsec;

}
