
;	extract_aux_data.pro,v 1.1 2003/08/28 07:08:35 ppq Exp	
;+
; NAME: extract_aux_data.pro
;
; PURPOSE: Extract EDI_PISO auxilliary file data to be used for
;          Cluster/EDI data analysis
;
; CATEGORY: Data extraction from an ASCII file
;
; CALLING SEQUENCE: An example:
;   extract_aux_data,'20010204',3,energy=0.5,stat=stat,t_ssm=t,efield_gse=e
;
; INPUTS: 
;   date             'YYYYMMDD'                            IN
;   sc_number        1, 2, 3 or 4                          IN
;
; KEYWORD PARAMETERS:  There are many!!!!
;
;        ----------------------------------------------
;        FILE-SPECIFYING FRONT-END CONTROL KEYWORDS, IN
;        ----------------------------------------------
;   filename=var_in     Full path+filename to an edi_piso aux file   IN
;                       No other file specifying keywords are examined
;
;   If you don't use the filename keyword, here are the rules for
;   using the other 'F'ile 'S'pecifying keywords which are defined
;   below, and are all indicated with an '(FS)':
;
;   EDI_PISO aux file names are assumed to have the following form (e.g.):
;
;     ./aux_data/20010223_1_164500_v1_30_DM00_p8_6_p29_3_aux.d
;
;   which is broken up as:
;
;     aux_path = './aux_data/'
;     date = '20010223'
;     sc_number = '1'
;     tstr_in = '164500'
;     version_in = 'v1_30'
;     dmstr_in = 'DM00_p8_6_p29_3'
;
;   The date and sc_number are input arguments to this routine.  The
;   rest are keywords.  Here are the defaults for these 4 keywords,
;   plus two more that are related:

;   aux_path=var_in (FS)  edi_piso aux file path           IN
;                         DEFAULT:
;                         aux_path=getenv('EDI_PISO_AUX_PATH')
;                         If this hasn't been set, then
;                         aux_path='./aux_data/'
;
;   /search_for_all (FS) Set this and all possible files with the
;                       correct date and spacecraft which are on the
;                       correct path (aux_path) will be searched 
;                       for, and the user can choose which one they
;                       want. This keyword over-rides ALL other
;                       file-specifying (FS) keywords except aux_path.
;
;   tstr_in=var_in (FS)  String, of form, e.g., '173000' which indicates
;                        the time string of the filename to search
;                        for.  In this example case, we're searching
;                        for hour=17, minute=30, second=00.
;                        DEFAULT: tstr_in = '*'
;
;   version_in=var_in (FS)  String, of form 'vN_NN' which indicates the
;                       version of the edi_piso aux file to search
;                       for.  DEFAULT: version_in = '*' (i.e. look for
;                       all versions and let user choose if there
;                       is more than one)

;   dmstr_in=var_in (FS)  String, of form, e.g.,  'DM00_p8_6_p29_3'
;                       which denotes the exact difference map string
;                       (dmstr) to search for.
;                       DEFAULT: dmstr_in = '*'
;
;   nchunk=var_in (FS) Choose the # of sub-intervals    IN
;                      analyzed per spin:
;                            1 = Spin resolution (4 sec, default)
;                            2 = 1/2 spin resolution (2 sec)
;                            4 = 1/4 spin resolution (1 sec)
;                      This keyword will over-ride the dmstr_in
;                      keyword by setting the dmstr_in search string
;                      to be '*p1_'+string(nchunk,'(i1)')+'_*'
;
;      DEFAULT: Example for date='20010223' and sc_number='1':
;         search_string = ./aux_data/20010223_1_*_*_*_aux.*
;
;        --------------------------------------------------
;        NON-FILE-SPECIFYING FRONT-END CONTROL KEYWORDS, IN
;        --------------------------------------------------
;
;   method=var_in       Choose which analysis method:       IN
;                        'TRI' = Return Triangulation results only
;                        'TOF' = Return Poorman's ToF results only
;                        'SMT' = Return Simultan ToF results only
;                        'RMT' = Return Richman's ToF results only
;                        'ALL' = Return ALL results (DEFAULT)
;
;   trange=var_in       2-D vector of form [h1,h2] specifying the
;                       start and end times (in hours) of the desired
;                       extraction. This can be useful if you know the
;                       specific time interval you want to analyze.
;                       Default: trange = [0,24] 
;
;   energy=var_in       Beam energy in keV (Default=1.0)    IN
;
;   /nobeam             Set this and no beam-related variables  IN
;                       will be available (it's a space-saver)
;
;   /nostatfilter       Filtering out fit failures according to   IN
;                       the value of methstat (see below) is
;                       NOT performed.  The user must examine
;                       methstat themselves to determine which 
;                       fits are valid.
;
;   ntime_max=var_in    Allows user to over-ride the default  IN
;                       maximum time dimension value.  The default
;                       value is now actually determined from the file
;                       itself, so this keyword is obsolete.
;
;   nbeam_max=var_in    Allows user to over-ride the default  IN
;                       maximum beams/per/interval value (100)
;
;            ----------------------
;            EXTRACTION STATUS, OUT
;            ----------------------
;   stat=var_out        0=extraction failed, 1=success      OUT
;   version_out=var_out Version of the edi_piso aux file used   OUT
;
;            ------------------------
;            BOOK-KEEPING/TIMING, OUT
;            ------------------------
;   ntime = var_out           # of intervals, scalar
;   t1 = var_out              Start time string, DIM=[ntime]
;   tmid = var_out            Mid-time string, DIM=[ntime]
;   t2 = var_out              Stop time string, DIM=[ntime]
;   t_ssm = var_out           Mid-time, sec. since midnight, DIM=[ntime]
;   twidth_sec = var_out      Interval width, seconds, DIM=[ntime]
;   bfield_scs = var_out      B, SCS, nT, DIM=[3,ntime]
;   bfield_gse = var_out      B, GSE, nT, DIM=[3,ntime]
;   vsc_gse = var_out         SC velocity, km/s
;   bvar1 = var_out           B-field total variance (reported by FGM)
;   bvar2 = var_out           B-field mag. variance (reported by FGM)
;   nnn = var_out             Parameter used in chip-length def., DIM=[ntime]
;   mm = var_out              2-bit EDI hardware parameter, DIM=[ntime]
;   code_type = var_out       Code type (0=short, 1=long), DIM=[ntime]
;   nbeam_usd = var_out       # of beams used by the chosen analysis
;                                method for each interval
;                                (Depending on the code version,
;                                there could be outlier removal or
;                                other reasons why nbeam_usd is not
;                                equal to nbeam_tot)
;   nbeam_tot = var_out       Total # of beams for each interval
;   flip = var_out            Used for bpp2scs reconstruction (ppq)
;   bpp2scs = var_out         Rotation matricies which allow
;                             transformation between the 'BPP', 
;                             (i.e. B-perp plane) coordinates and
;                             the SCS coordinates.
;
;  IMPORTANT ROTATION MATRIX NOTICE:  
;        The rotation matrix bpp2scs is stored in the
;        traditional physics/math sense of (irow,icol).  So that to
;        rotate a vector:
;             vprime(i) = M(i,k)*v(k)  (Einstein notation, sum over k)
;        This is contrary to the fact that IDL is row major and that
;        matricies are assumed to be stored as (icol,irow).  So,
;        BE CAREFUL WHEN YOU USE IDL MATRIX OPERATORS!
;
;   scs2bpp_refvec = var_out  Unit vector, DIM=[3,ntime], used in the
;                             bpp2scs re-construction
;   rotmat=rotmat             Set this keyword if you want
;                             bpp2scs to be available.
;                             The default is NOT to have
;                             bpp2scs available because it could be
;                             rather large.
;
;   ----------------------------------------------------------------------
;   METHOD INDEPENDENT RESULTS (Triangulation, Poorman's ToF, Simultan ToF...)
;                        DIM=[ntime], OUT
;   ----------------------------------------------------------------------
;   meth_usd = var_out          Which analysis method was ultimately used:
;                                  0 = Triangulation
;                                  1 = Poorman's ToF
;                                  2 = Simultan ToF
;                                  3 = Richman's ToF
;
;   target_mag = var_out        Target magnitude, meters
;   error_target_mag = var_out  
;               TRIANGULATION:  Error calculated from actual chi2
;                               surface using standard dchi2 increment
;                               for 90% confidence (which is
;                               dchi2=4.61 for nparm=2).  For a
;                               discussion of this technique, see
;                               "Numerical Recipes" by Press et al.,
;                               Chapter 14 (or, the 'Modeling of Data'
;                               chapter) and the section 'Confidence
;                               Limits on Estimated Model Parameters'.
;                               See below a description of 
;                               error2_target_mag which uses a scaled dchi2.
;
;               POORMAN'S TOF:  Hans Vaith's 'TofError.pro' used
;               SIMULTAN TOF:   Same as Poorman's ToF
;               RICHMAN'S TOF:  Same as Poorman's ToF
;
;   target_azi = var_out        Target azimuth, degrees, in BPP
;   error_target_azi = var_out
;               TRIANGULATION:  See above description for
;                               error_target_mag
;               POORMAN'S TOF:  Angular width of beams as reported
;                               from the parallelism test, degrees
;               SIMULTAN TOF:   Same as Poorman's ToF
;               RICHMAN'S TOF:  Same as Poorman's ToF
;
;   tgyro = var_out             Target gyrotime, microsec, calculated
;                               from the magnitude of the average
;                               magnetic field.
;
;   methstat = var_out          Indicates the success or failure of the
;                               result.  If you don't set the
;                               'nostatfilter' keyword, only
;                               successful results are returned and
;                               methstat will indicate this fact by
;                               being all zeros (for v1_29 or higher).
;                               v1_28 or less:    1        =  success
;                                           anything else  =  fail
;                               v1_29 or higher:  0        =  success
;                                           anything else  =  fail
;
;   sbytes = var_out            DIM=[7,ntime] = Status byte array that
;                               is output to the IFF file.  For a
;                               description of these bytes see:
;                               STATUS_BYTE_DEF.txt
;    NOTE: sbytes(1,*) and sbytes(2,*) are not currently output correctly
;    because of how the aux data is written.  They are defined and
;    output to the IFF file, but NOT HERE!  (because they involve
;    statistics for the whole spin regardless of beam energy, and the
;    aux data is output as the code passes over each energy...)
;
;   -----------------------------------------------------
;   METHOD DEPENDENT RESULTS -- Triangulation, DIM=[ntime], OUT
;    ----------------------------------------------------
;   rchi2_cg = var_out          TRI reduced chi-squared of coarse grid
;                                  fit after outlier removal (if any)
;   clevel_cg = var_out         TRI confidence level associated with
;                               rchi2_cg; Probability of observing chi-squared
;                               value LARGER than rchi2_cg*ndegfree_cg for a
;                               random sample of N observations w/ ndegfree_cg
;                               degrees of freedom, i.e.
;                               clevel_cg = 0 means you can't have much
;                                        confidence in your minimization
;                                        since a random realization of the
;                                        data will definitely produce a
;                                        lower chi2_cg value than your result.
;                               clevel_cg = 1 means that you can be 100%
;                                        confident of your minimization
;                                        since any random realization of
;                                        the data will have a higher chi2_cg
;                                        value than your result.
;   rchi2_fg = var_out          Same as rchi2_cg, but for the fine grid
;   clevel_fg = var_out         Same as clevel_cg, but for the fine grid
;   error2_target_mag = var_out Error calculated from actual chi2
;                               surface using 90% confidence dchi2
;                               increment scaled by the value of
;                               rchi2_fg (see above).  This is the
;                               same as using a normalized chi2
;                               surface (where rchi2_min=1.) and going
;                               up the standard dchi2 to define the
;                               error extent.
;
;   error2_target_azi = var_out
;   error2_tgyro = var_out
;
;   -----------------------------------------------------------
;   METHOD DEPENDENT RESULTS -- Poorman's ToF, DIM=[ntime], OUT
;                               Richman's ToF too
;   -----------------------------------------------------------
;          OF THOSE BEAMS USED (see nbeam_usd):
;   nbeam_classA_tot = var_out      how many are Class "A"
;   nbeam_nonA_tot = var_out        how many are not Class "A"
;   nbeam_classA_towards = var_out  how many are Class "A" and towards
;   nbeam_classA_away = var_out     how many are Class "A" and away
;   nbeam_nonA_towards = var_out    how many are not Class "A", but towards
;   nbeam_nonA_away = var_out       how many are not Class "A", but away
;
;   -----------------------------------------------------------
;   DERIVED PARAMETERS (method independent), DIM=[3,ntime], OUT
;   -----------------------------------------------------------
;   drift_step_scs = var_out      meters
;   drift_velo_scs = var_out      km/s
;   efield_scs = var_out          mV/m
;   drift_step_gse = var_out      meters
;   drift_velo_gse = var_out      km/s, Corrected for SC motion
;   efield_gse = var_out          mV/m, Corrected for SC motion
;   drift_velo_nocorr_gse = var_out km/s, NOT corrected for SC motion
;   efield_nocorr_gse = var_out     mV/m, NOT corrected for SC motion
;
;   ------------------------------------------------------------------
;   BEAM QUANTITIES, DIM=[nbeam_max,ntime], OUT
;   NOTE: For interval i, all of these 'beam' arrays only have valid
;   values for the following indices:  (0:nbeam_tot(i)-1,i).
;   For example, out_beam(0:nbeam_tot(i)-1,i) will have values ranging
;   from 0 to 6, but out_beam(nbeam_tot(i):nbeam_max-1,i) will all be
;   fill values (255 in this case because out_beam is an integer)
;   ------------------------------------------------------------------
;   out_beam = var_out             0 and 1 = Used by analysis method
;                                        2 = Not used, TRI outlier
;                                        3 = Not used, class restriction
;                                        4 = Not used, unworthy of PMT
;                                        5 = Not used, unworthy of SMT
;                                        6 = Not used, unworthy of RMT
;
;   qual_beam = var_out            Beam quality, 1,2 or 3
;   bestord_beam = var_out         Beam order as assigned by the chosen
;                                        analysis method
;
;   class_init_beam = var_out      Initial beam class (A, B, C)
;                  'A' = Definitive assignment by RunEst
;                  'B' = Choices of order given by RunEst
;                  'C' = No order info given by RunEst
;   class_beam = var_out           Final beam class (A, B, C or R=reassigned)
;   gunid_beam = var_out           Gun ID (1 or 2)
;   xg_beam = var_out              Xgun, BPP-coord., meters
;   yg_beam = var_out              Ygun, BPP-coord., meters
;   alpha_beam = var_out           Fire-ang (rad), BPP-coord.
;   tof_beam = var_out             Time-of-flight, microsec
;   atime_ssm_beam = var_out       atime, sec. since midnight
;   btime_ssm_beam = var_out       beam_time, sec. since midnight
;   gdotb_beam = var_out           acos(Firing-dir. dot B), deg.
;   maxchan_beam = var_out         Channel # (should be 6,7 or 8)
;   bwidth_beam = var_out          Beam width (proj. in B-perp plane), degrees
;
; COMMON BLOCKS:  extract_aux_data_cb
;                 extract_aux_data_cb_internal
;
; MODIFICATION HISTORY:
;   Created February, 2001 by Pamela A. Puhl-Quinn, ppq@mpe.mpg.de
;
; NOTES:
; NOTE on SCS system: 
;            Z_SCS is the spin
;            axis, X_SCS instantaneously set along the
;            Sun Sensor axis when the Sun Reference
;            Pulse is generated; this is the system
;            used by FGM.
;
; NOTE on BPP coordinates: B-perp plane
;            Z_BPP is either along or opposed to B
;            Z_BPP has same sense as Z-GSE
;            X_BPP has same sense as X-SCS (sunward)
;            Y_BPP completes system
;
;-
;==================================================='

pro extract_aux_data, date, sc_number, $
                      aux_path=aux_path_in, $
                      tstr_in=tstr_in, $
                      filename=filename_in, $
                      version_in=version_in, $
                      dmstr_in=dmstr_in, $
                      energy=energy_in, $
                      search_for_all=search_for_all, $
                      trange=trange_in, $
                      $
                      nchunk=nchunk_in, $
                      method=method_in, $
                      $
                      nobeam=nobeam, $
                      nostatfilter=nostatfilter, $
                      stat=stat_out, $
                      version_out=version_out, $
                      rfill=rfill_out, $
                      ntime_max=ntime_max_in, $
                      nbeam_max=nbeam_max_in, $
                      fname_out=fname_out, $
                      dmstr_out=dmstr_out, $
                      epdm_str=epdm_str, $
                      $
                      ntime = ntime_out, $
                      t1 = t1_out, $
                      tmid = tmid_out, $
                      t2 = t2_out, $
                      t_ssm = t_ssm_out, $
                      twidth_sec = twidth_sec_out, $
                      bfield_scs = bfield_scs_out, $
                      bfield_gse = bfield_gse_out, $
                      scs2bpp_refvec = scs2bpp_refvec_out, $
                      rmax=rmax_out, $
                      r_max_2=r_max_2_out, $
                      bvar1 = bvar1_out, $
                      bvar2 = bvar2_out, $
                      nnn = nnn_out, $
                      mm = mm_out, $
                      code_type = code_type_out, $
                      nbeam_tot = nbeam_tot_out, $
                      $
                      $         ; Stats on those beams used
                      nbeam_usd = nbeam_usd_out, $
                      nbeam_classA_tot = nbeam_classA_tot_out, $
                      nbeam_nonA_tot = nbeam_nonA_tot_out, $
                      nbeam_classA_towards = nbeam_classA_towards_out, $
                      nbeam_classA_away = nbeam_classA_away_out, $
                      nbeam_nonA_towards = nbeam_nonA_towards_out, $
                      nbeam_nonA_away = nbeam_nonA_away_out, $
                      $
                      flip = flip_out, $
                      bpp2scs = bpp2scs_out, $
                      rotmat=rotmat, $
                      $
                      target_mag = target_mag_out, $
                      error_target_mag = error_target_mag_out, $
                      error2_target_mag = error2_target_mag_out, $
                      target_azi = target_azi_out, $
                      error_target_azi = error_target_azi_out, $
                      error2_target_azi = error2_target_azi_out, $
                      tgyro = tgyro_out, $
                      error_tgyro = error_tgyro_out, $
                      error2_tgyro = error2_tgyro_out, $
                      methstat = methstat_out, $
                      sbytes = sbytes_out, $
                      meth_usd = meth_usd_out, $
                      comptime = comptime_out, $
                      ambig_180 = ambig_180_out, $
                      ptest_mean_angle = ptest_mean_angle_out, $
                      ptest_stdev_angle = ptest_stdev_angle_out, $
                      $
                      rchi2_cg = rchi2_cg_out, $
                      rchi2_unpen_cg = rchi2_unpen_cg_out, $
                      clevel_cg = clevel_cg_out, $
                      rchi2_fg = rchi2_fg_out, $
                      rchi2_unpen_fg = rchi2_unpen_fg_out, $
                      clevel_fg = clevel_fg_out, $
                      $
                      drift_step_scs = drift_step_scs_out, $
                      drift_velo_scs = drift_velo_scs_out, $
                      efield_scs = efield_scs_out, $
                      drift_step_gse = drift_step_gse_out, $
                      drift_velo_gse = drift_velo_gse_out, $
                      efield_gse = efield_gse_out, $
                      drift_velo_nocorr_gse = drift_velo_nocorr_gse_out, $
                      efield_nocorr_gse = efield_nocorr_gse_out, $
                      vsc_gse = vsc_gse_out, $
                      $
                      beam_load_in=beam_load_in, $
                      out_beam = out_beam_out, $
                      penalty_beam = penalty_beam_out, $
                      used_beam = used_beam_out, $
                      bestord_beam = bestord_beam_out, $
                      gunid_beam = gunid_beam_out, $
                      xg_beam = xg_beam_out, $
                      yg_beam = yg_beam_out, $
                      alpha_beam = alpha_beam_out, $
                      tof_beam = tof_beam_out, $
                      atime_ssm_beam = atime_ssm_beam_out, $
                      btime_ssm_beam = btime_ssm_beam_out, $
                      gdotb_beam = gdotb_beam_out, $
                      maxchan_beam = maxchan_beam_out, $
                      bwidth_beam = bwidth_beam_out, $
                      qual_beam = qual_beam_out, $
                      class_beam = class_beam_out, $
                      class_init_beam = class_init_beam_out, $
                      $
                      envar_path=envar_path

common extract_aux_data_cb, ntime,t1,tmid,t2,t_ssm,twidth_sec,bfield_scs,bfield_gse,scs2bpp_refvec,rmax,r_max_2,bvar1,bvar2,nnn,mm,code_type,nbeam_usd,nbeam_tot,flip,target_mag,target_azi,error_target_mag,error_target_azi,error2_target_mag,error2_target_azi,tgyro,error_tgyro,error2_tgyro,rchi2_cg,rchi2_unpen_cg,clevel_cg,rchi2_fg,rchi2_unpen_fg,clevel_fg,methstat,sbytes,drift_step_scs,drift_velo_scs,efield_scs,drift_step_gse,drift_velo_gse,efield_gse,drift_velo_nocorr_gse,efield_nocorr_gse,vsc_gse,out_beam,class_beam,penalty_beam,used_beam,class_init_beam,bestord_beam,gunid_beam,xg_beam,yg_beam,alpha_beam,tof_beam,atime_ssm_beam,btime_ssm_beam,gdotb_beam,maxchan_beam,bwidth_beam,qual_beam,meth_usd, comptime, ambig_180, ptest_mean_angle, ptest_stdev_angle, nbeam_classA_tot, nbeam_nonA_tot, nbeam_classA_towards, nbeam_classA_away, nbeam_nonA_towards, nbeam_nonA_away, ismt, edi3_smt

common extract_aux_data_cb_internal, cfname, cenergy, cnobeam, cbeam_load, cmethod, cnostatfilter, ctrange

if n_params() eq 0 then begin
    doc_library, 'extract_aux_data'
    return
endif

rfill = -1e+31
stat_out=0                      ; Fail
sc_str = string(sc_number,'(i1)')
date_str = date

;=========================
; FILE-SPECIFYING KEYWORDS
;=========================
if (n_elements(filename_in) ne 0) then begin
    f = findfile(filename_in)
    if (f(0) eq '') then begin
        message, 'File not found: '+filename_in, /cont
        return
    endif else begin
        fname=filename_in
    endelse
endif else begin
    ident_str = '_aux.'
    config_str = '*'
    
    if (n_elements(aux_path_in) eq 0) then begin
        aux_path = getenv('EDI_PISO_AUX_PATH')
        if (aux_path eq '') then aux_path = './aux_data/'
    endif else begin
        aux_path = aux_path_in
    endelse
    
    if keyword_set(search_for_all) then begin
        tstr = '*'
        vers_str = '*'
        dmstr = '*'
    endif else begin
        if (n_elements(tstr_in) ne 0) then tstr = tstr_in else tstr = '*'
        if (n_elements(version_in) ne 0) then vers_str = version_in else $
          vers_str = '*'
        if (n_elements(dmstr_in) ne 0) then dmstr = dmstr_in else $
          dmstr = '*'
        if (n_elements(nchunk_in) ne 0) then dmstr = '*p1_'+$
          string(nchunk_in,'(i1)')+'*'
    endelse
    
; Search string
    ss = aux_path+'/'+date_str+'_'+sc_str+'_'+tstr+'_'+vers_str+'_'+ $
      dmstr+ident_str+config_str
    
    f = findfile(ss)
    if (f(0) eq '') then begin
        message, 'Aux file not found; Path searched: ', /cont
        message, ss, /cont
        return
    endif else begin
        if (n_elements(f) gt 1) then begin
            res=''
            message, 'More than one file found: ',/cont
            for i=0,n_elements(f)-1 do begin
                print, '  '+strtrim(i,2)+'  '+f(i)
            endfor
            read, res, prompt='Which file do you want? '
            fname = f(long(res))
        endif else begin
            fname = f(0)
        endelse
    endelse
endelse

; Output info from filename
; version_out, dmstr_out
pos = strpos(fname,'edi_piso_aux')
if (pos(0) eq -1) then begin
    pos = strpos(fname,'_EPDM_')
    if (pos(0) ne -1) then begin
        version_out = strmid(fname,pos-5,5)
        pos2 = strpos(fname,'_aux.')
        dmstr_out = strmid(fname,pos+1,pos2-pos-1)
    endif else begin
        pos = strpos(fname,'_DM00_')
        if (pos(0) ne -1) then begin
            version_out = strmid(fname,pos-5,5)
            pos2 = strpos(fname,'_aux.')
            dmstr_out = strmid(fname,pos+1,pos2-pos-1)
        endif else begin
            pos = strpos(fname,'_DM01_')
            if (pos(0) ne -1) then begin
                version_out = strmid(fname,pos-5,5)
                pos2 = strpos(fname,'_aux.')
                dmstr_out = strmid(fname,pos+1,pos2-pos-1)
            endif else begin
                pos = strpos(fname,'_DM02_')
                if (pos(0) ne -1) then begin
                    version_out = strmid(fname,pos-5,5)
                    pos2 = strpos(fname,'_aux.')
                    dmstr_out = strmid(fname,pos+1,pos2-pos-1)
                endif else begin
                    pos = strpos(fname,'_DM03_')
                    if (pos(0) ne -1) then begin
                        version_out = strmid(fname,pos-5,5)
                        pos2 = strpos(fname,'_aux.')
                        dmstr_out = strmid(fname,pos+1,pos2-pos-1)
                    endif else begin
                        version_out = 'unkno'
                        dmstr_out = 'unknown'
                    endelse
                endelse
            endelse
        endelse
    endelse
endif else begin
    version_out = strmid(fname,pos+15,5)
    dmstr_out = 'no_epdm'
endelse

; Fudge for backwards compatability
epdm_str = dmstr_out

message, 'File found: '+fname, /cont

;=============================
; NON-FILE-SPECIFYING KEYWORDS
;=============================
if (n_elements(trange_in) ne 0) then trange=trange_in else trange=[0,24]
if (n_elements(method_in) ne 0) then method = method_in else method = 'ALL'
nobeam = keyword_set(nobeam)
if (nobeam) then beam_load = [1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0] else begin $
  if (n_elements(beam_load_in) ne 0) then beam_load = beam_load_in else $
  beam_load = [1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]
endelse
beam_load(0) = 1
beam_load(1) = 1
nostatfilter = keyword_set(nostatfilter)
if (n_elements(energy_in) eq 0) then energy=1.0 else energy=float(energy_in)
if (energy ne 1.0 and energy ne 0.5) then begin
    message, '"energy" keyword has invalid value', /cont
    message, 'Must choose either energy=1.0 or energy=0.5', /cont
    return
endif
if (n_elements(nbeam_max_in) eq 0) then nbeam_max = 100 else nbeam_max = $
  nbeam_max_in

; Try to be smart about ntime_max
if (n_elements(ntime_max_in) eq 0) then begin
    date_vax = date2datevax(date)

;Not good for MFO:    spawn, 'grep "'+date_vax+'" '+fname+' | wc -l', sout
;Doesn't work at all: spawn, ['grep','"'+date_vax+'"',fname,'|','wc','-l'], $
;                        sout, /noshell
    
    spawn, ['wc','-l',fname], sout, /noshell
    ipos = strpos(sout(0),fname)
    if (ipos eq -1) then ntime_max = long(sout(0)) else $
      ntime_max = long(strmid(sout(0),0,ipos-1))
    
    ntime_max = long ( (ntime_max-60.)/17. ) + 10
    
endif else ntime_max = ntime_max_in

if (version_out ne 'devel') then begin
    vers = float(strmid(version_out,1,1))
    vers = vers + float(strmid(version_out,3,2))/100.
endif else vers = 9.99

if (n_elements(cfname) eq 0) then goto, readit
if (cfname ne fname) then goto, readit
if (cenergy ne energy) then goto, readit
if (cnobeam ne nobeam) then goto, readit
if (total(abs(cbeam_load - beam_load)) ne 0) then goto, readit
if (cmethod ne method) then goto, readit
if (ctrange(0) ne trange(0)) then goto, readit
if (ctrange(1) ne trange(1)) then goto, readit
if (cnostatfilter ne nostatfilter) then goto, readit
goto, already_in_buffer

;==============================================================
;==============================================================
readit:

print, 'Opening: '+fname
openr, unit, fname, /get_lun

;==============HEADER INFO====================
dum = ''

if (vers le 1.28) then begin
    for i=0,25 do begin
        readf, unit, dum
        print, dum
    endfor
    readf, unit, dum & print, dum
    readf, unit, dum & print, dum
    readf, unit, dum & print, dum
    readf, unit, dum & print, dum
endif else begin
    goon = 1
    while goon do begin
        readf, unit, dum
        print, dum
        if strmid(dum,0,1) ne '#' then goon = 0
    endwhile
    readf, unit, dum & print, dum
endelse

print, 'Done with header read'

;==============VARIABLE ALLOCATIONS===========
energyeV = fltarr(ntime_max) + rfill
status = intarr(ntime_max) + 255
meth_usd = intarr(ntime_max) + 255
comptime = fltarr(ntime_max) + rfill
sbytes = intarr(7,ntime_max) + 255

t1 = strarr(ntime_max)
t2 = strarr(ntime_max)
bfield_scs = fltarr(3,ntime_max)
bfield_gse = fltarr(3,ntime_max)+rfill
scs2bpp_refvec = fltarr(3,ntime_max)+rfill
rmax = fltarr(ntime_max)+rfill
r_max_2 = fltarr(ntime_max)+rfill
bvar1 = fltarr(ntime_max)
bvar2 = fltarr(ntime_max)
nnn = intarr(ntime_max)
mm = intarr(ntime_max)
code_type = intarr(ntime_max)
nbeam_usd = intarr(ntime_max)
nbeam_tot = intarr(ntime_max)
flip = bytarr(ntime_max)

target_mag = fltarr(ntime_max)
error_target_mag = fltarr(ntime_max)
error2_target_mag = fltarr(ntime_max)
target_azi = fltarr(ntime_max)
error_target_azi = fltarr(ntime_max)
error2_target_azi = fltarr(ntime_max)
tgyro = fltarr(ntime_max)
error_tgyro = fltarr(ntime_max)
error2_tgyro = fltarr(ntime_max)

drift_step_scs = fltarr(3,ntime_max)
drift_velo_scs = fltarr(3,ntime_max)
efield_scs = fltarr(3,ntime_max)
drift_step_gse = fltarr(3,ntime_max)
drift_velo_gse = fltarr(3,ntime_max)
efield_gse = fltarr(3,ntime_max)
drift_velo_nocorr_gse = fltarr(3,ntime_max)
efield_nocorr_gse = fltarr(3,ntime_max)
vsc_gse = fltarr(3,ntime_max)

nbeam_classA_tot = intarr(ntime_max)+255
nbeam_nonA_tot  = intarr(ntime_max)+255
nbeam_classA_towards = intarr(ntime_max)+255
nbeam_classA_away = intarr(ntime_max)+255
nbeam_nonA_towards = intarr(ntime_max)+255
nbeam_nonA_away = intarr(ntime_max)+255

ismt = intarr(ntime_max)
edi3_smt = fltarr(3,ntime_max)+rfill

; These two beam quantities are needed for nbeam_usd determination
out_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 0
used_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 1

if (beam_load(2)) then class_beam = make_array(nbeam_max,ntime_max,/str,value='F') ; 2
if (beam_load(3)) then penalty_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 3
if (beam_load(4)) then class_init_beam = make_array(nbeam_max,ntime_max,/str,value='F') ; 4
if (beam_load(5)) then maxchan_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 5
if (beam_load(6)) then qual_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 6
if (beam_load(7)) then bestord_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 7   
if (beam_load(8)) then gunid_beam = make_array(nbeam_max,ntime_max,/int,value=255) ; 8
if (beam_load(9)) then xg_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 9
if (beam_load(10)) then yg_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 10
if (beam_load(11)) then alpha_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 11
if (beam_load(12)) then tof_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 12
if (beam_load(13)) then atime_ssm_beam = make_array(nbeam_max,ntime_max,/double,value=rfill) ; 13
if (beam_load(14)) then btime_ssm_beam = make_array(nbeam_max,ntime_max,/double,value=rfill) ; 14
if (beam_load(15)) then gdotb_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 15
if (beam_load(16)) then bwidth_beam = make_array(nbeam_max,ntime_max,/float,value=rfill) ; 16

rchi2_cg = fltarr(ntime_max)
rchi2_unpen_cg = fltarr(ntime_max)
clevel_cg = fltarr(ntime_max)
rchi2_fg = fltarr(ntime_max)
rchi2_unpen_fg = fltarr(ntime_max)
clevel_fg = fltarr(ntime_max)

ambig_180 = intarr(ntime_max)
ptest_mean_angle = fltarr(ntime_max)
ptest_stdev_angle = fltarr(ntime_max)

ii=-1L
while not eof(unit) do begin
    
    if (ii ge ntime_max-1) then message, 'Array size for time-dimension not big enough; Must set ntime_max keyword to be larger than : '+strtrim(ntime_max,2)
    
    lbstr = ''
    rbstr = ''
    bx_scs = 0.
    by_scs = 0.
    bz_scs = 0.
    bx_gse = 0.
    by_gse = 0.
    bz_gse = 0.
    bv1 = 0.
    bv2 = 0.
    n = 0
    m = 0
    yestof_whatech = 0
    ct = 0
    nb = 0
    erg = 0.
    fli = 0
    rvx = 0.
    rvy = 0.
    rvz = 0.
    rmx = 0.
    rmx2 = 0.
    readf, unit, dum

; Skip this record if the center time is not within the specified trange
;24-Apr-2001 09:59:58.025365
    th = float(strmid(dum,12,2)) + $
      float(strmid(dum,15,2))/60. + $
      float(strmid(dum,18,9))/3600. + 2./3600.

    print, trange(0), th, trange(1)

    if (th lt trange(0)) then begin
        readf, unit, dum        ; gunid_beam
        readf, unit, dum        ; xg_beam
        readf, unit, dum        ; yg_beam
        readf, unit, dum        ; alpha_beam
        readf, unit, dum        ; tof_beam
        readf, unit, dum        ; atime_ssm_beam
        readf, unit, dum        ; btime_ssm_beam
        readf, unit, dum        ; gdotb_beam
        readf, unit, dum        ; maxchan_beam
        if (vers gt 1.21) then readf, unit, dum ; bwidth_beam
        if (vers gt 1.28) then readf, unit, dum ; qual_beam
        if (vers le 1.28) then readf, unit, dum ; toaw_beam
        readf, unit, dum        ; Results
        readf, unit, dum        ; out_beam
        readf, unit, dum        ; bestord_beam
        if (vers gt 1.23 and vers le 1.28) then readf, unit, dum ; preord_beam
        if (vers gt 1.28) then readf, unit, dum ; class_init_beam
        if (vers gt 1.28) then readf, unit, dum ; class_beam
        if (vers gt 2.27) then readf, unit, dum ; penalty_beam
        if (vers gt 2.27) then readf, unit, dum ; used_beam
        goto, record_skipped
    endif

    if (th gt trange(1)) then goto, stop_reading
        
    if (vers le 1.20) then begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, yestof_whatech, ct, nb, erg, fli, $
          format= $
          '(2(a27,1x),5(e13.6,1x),3(i3,1x),i1,1x,i3,1x,f3.1,1x,i1)'
        ii = ii + 1
    endif else if (vers gt 1.20 and vers le 1.24) then begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, yestof_whatech, ct, nb, erg, fli, $
          bx_gse, by_gse, bz_gse, $
          format= $
          '(2(a27,1x),5(e13.6,1x),3(i3,1x),i1,1x,i3,1x,f3.1,1x,i1,1x,3(e13.6,1x))'
        ii = ii + 1
        bfield_gse(0:2,ii) = [bx_gse, by_gse, bz_gse]
    endif else if (vers gt 1.24 and vers le 1.28) then begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, yestof_whatech, ct, nb, erg, fli, $
          bx_gse, by_gse, bz_gse, rvx, rvy, rvz, $
          format= $
          '(2(a27,1x),5(e13.6,1x),3(i3,1x),i1,1x,i3,1x,f3.1,1x,i1,1x,3(e13.6,1x)3(e13.6,1x))'
        ii = ii + 1
        bfield_gse(0:2,ii) = [bx_gse, by_gse, bz_gse]
        scs2bpp_refvec(0:2,ii) = [rvx,rvy,rvz]
    endif else if (vers gt 1.28 and vers le 1.36) then begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, ct, nb, erg, fli, $
          bx_gse, by_gse, bz_gse, rvx, rvy, rvz, $
          format= $
          '(2(a27,1x),'+$
          '5(e13.6,1x),'+$
          '2(i3,1x),'+$
          'i1,1x,i3,1x,f3.1,1x,i1,1x,'+$
          '3(e13.6,1x),3(e13.6,1x))'
        ii = ii + 1
        bfield_gse(0:2,ii) = [bx_gse, by_gse, bz_gse]
        scs2bpp_refvec(0:2,ii) = [rvx,rvy,rvz]
    endif else if (vers gt 1.36 and vers le 2.27) then begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, ct, nb, erg, fli, $
          bx_gse, by_gse, bz_gse, rvx, rvy, rvz, $
          rmx, $
          format= $
          '(2(a27,1x),'+$
          '5(e13.6,1x),'+$
          '2(i3,1x),'+$
          'i1,1x,i3,1x,f3.1,1x,i1,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),e13.6)'
        ii = ii + 1
        bfield_gse(0:2,ii) = [bx_gse, by_gse, bz_gse]
        scs2bpp_refvec(0:2,ii) = [rvx,rvy,rvz]
        rmax(ii) = rmx
    endif else begin
        reads, dum, lbstr, rbstr, bx_scs, by_scs, bz_scs, bv1, bv2, $
          n, m, ct, nb, erg, fli, $
          bx_gse, by_gse, bz_gse, rvx, rvy, rvz, $
          rmx, rmx2, $
          format= $
          '(2(a27,1x),'+$
          '5(e13.6,1x),'+$
          '2(i3,1x),'+$
          'i1,1x,i3,1x,f3.1,1x,i1,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),e13.6,1x,e13.6)'
        ii = ii + 1
        bfield_gse(0:2,ii) = [bx_gse, by_gse, bz_gse]
        scs2bpp_refvec(0:2,ii) = [rvx,rvy,rvz]
        rmax(ii) = rmx
        r_max_2(ii) = rmx2
    endelse
    
    t1(ii) = lbstr
    t2(ii) = rbstr
    bfield_scs(0:2,ii) = [bx_scs, by_scs, bz_scs]
    bvar1(ii) = bv1
    bvar2(ii) = bv2
    nnn(ii) = n
    mm(ii) = m
    code_type(ii) = ct
    nbeam_tot(ii) = nb
    energyeV(ii) = erg
    flip(ii) = fli
;======================================================    
    nbstr = strtrim(nb,2)
    ls = '('+nbstr
    
    readf, unit, dum
    if (beam_load(8)) then begin
        if (nb ge nbeam_max) then message, 'Array size for ' +$
          'beam-dimension not big enough ; Must set nbeam_max ' + $
          'keyword to be larger than : '+strtrim(nbeam_max,2)        
        ddat = intarr(nb)
        reads, dum, ddat, format = ls+'(i1,1x))'
        gunid_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(9)) then begin
        ddat = fltarr(nb)
        reads, dum, ddat, format = ls+'(e13.6,1x))'
        xg_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(10)) then begin
        ddat = fltarr(nb)
        reads, dum, ddat, format = ls+'(e13.6,1x))'
        yg_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(11)) then begin
        ddat = fltarr(nb)
        reads, dum, ddat, format = ls+'(e13.6,1x))'
        alpha_beam(0:nb-1,ii) = ddat*!pi/180. ; radians
    endif
    
    readf, unit, dum
    if (beam_load(12)) then begin
        ddat = fltarr(nb)
        reads, dum, ddat, format = ls+'(e13.6,1x))'
        tof_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(13)) then begin
        ddat = dblarr(nb)
        reads, dum, ddat, format = ls+'(d20.13,1x))'
        atime_ssm_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(14)) then begin
        ddat = dblarr(nb)
        reads, dum, ddat, format = ls+'(d20.13,1x))'
        btime_ssm_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(15)) then begin
        ddat = fltarr(nb)
        reads, dum, ddat, format = ls+'(d20.13,1x))'
        gdotb_beam(0:nb-1,ii) = ddat
    endif
    
    readf, unit, dum
    if (beam_load(5)) then begin
        ddat = intarr(nb)
        reads, dum, ddat, format = ls+'(i2,1x))'
        maxchan_beam(0:nb-1,ii) = ddat
    endif
    
    if (vers gt 1.21) then begin
        readf, unit, dum
        if (beam_load(16)) then begin
            ddat = fltarr(nb)
            reads, dum, ddat, format = ls+'(f6.3,1x))'
            bwidth_beam(0:nb-1,ii) = ddat
        endif
    endif

    if (vers gt 1.28) then begin
        readf, unit, dum
        if (beam_load(6)) then begin
            ddat = intarr(nb)
            reads, dum, ddat, format = ls+'(i2,1x))'
            qual_beam(0:nb-1,ii) = ddat
        endif
    endif
    
;===========================================================
    
    readf, unit, dum
    stat = 0
    meth = 0
    ctime = 0.
    ytof_in = 0
    ytof_out = 0
    edi6 = fltarr(6)
    r2 = 0.
    r2unpen = 0.
    r2_fg = rfill
    r2unpen_fg = rfill
    cl = rfill
    cl_fg = rfill
    d_scs = fltarr(3)
    v_scs = fltarr(3)
    e_scs = fltarr(3)
    d_gse = fltarr(3)
    v_gse = fltarr(3)
    e_gse = fltarr(3)
    vgsenocorr = fltarr(3)+rfill
    egsenocorr = fltarr(3)+rfill
    vscgse = fltarr(3)+rfill
    rderr2 = rfill
    pderr2 = rfill
    tgerr2 = rfill
    amb180 = 0
    ptest_mean = 0.
    ptest_stdev = 0.
    s0 = 0
    s1or2 = 0
    s3 = 0
    s4 = 0
    s5 = 0
    s6 = 0
    cat = 0
    caa = 0
    nat = 0
    naa = 0
    jsmt = 0
    e3smt0 = rfill
    e3smt1 = rfill
    e3smt2 = rfill
    
    if (vers eq 1.22) then begin
        reads, dum, stat, ctime, ytof_in, ytof_out, $
          edi6, r2, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, cl, $
          format=$
          '(i3,1x,e13.6,1x,2(i3,1x),6(e13.6,1x),e13.6,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),e13.6)'
    endif else if (vers eq 1.23) then begin
        reads, dum, stat, ctime, ytof_in, ytof_out, $
          edi6, r2, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, cl, $
          r2_fg, cl_fg, $
          format=$
          '(i3,1x,e13.6,1x,2(i3,1x),6(e13.6,1x),e13.6,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),3(e13.6,1x))'
    endif else if (vers gt 1.23 and vers le 1.28) then begin
        
;                reads, dum, stat, ctime, ytof_in, ytof_out, $
;                  edi6, r2, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, cl, $
;                  r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
;                  format=$
;                  '(i3,1x,e13.6,1x,2(i3,1x),6(e13.6,1x),e13.6,1x,'+$
;                  '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
;                  '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
;                  '3(e13.6,1x),3(e13.6,1x))'
        
        reads, dum, stat, ctime, ytof_in, ytof_out, $
          edi6, r2, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, cl, $
          r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          format=$
          '(i3,1x,e13.6,1x,2(i3,1x),6(e13.6,1x),'+$
          'e13.6,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x))'
        
    endif else if (vers lt 1.22) then begin
        reads, dum, stat, ctime, ytof_in, ytof_out, $
          edi6, r2, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, format=$
          '(i3,1x,e13.6,1x,2(i3,1x),6(e13.6,1x),e13.6,1x,'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x))'
    endif else if (vers gt 1.28 and vers le 1.36) then begin

;============================
; Quick fix on format problem
        ipos = strpos(dum,'***')
        if (ipos ne -1) then begin
            head = strmid(dum,0,ipos)
            tail = strmid(dum,ipos+3)
            dum = head+'254'+tail
        endif
;============================

        reads, dum, stat, meth, ctime, $
          edi6, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, r2, cl, $
          r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          amb180, ptest_mean, ptest_stdev, $
          s0,s1or2,s3,s4,s5,s6, $
          format=$
          '(i3,1x,i3,1x,e13.6,1x,'+$
          '6(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),i3,1x,'+$
          '2(e13.6,1x),'+$
          '3(i3,1x),'+$
          '3(i3,1x))'
        ambig_180(ii) = amb180
        ptest_mean_angle(ii) = ptest_mean
        ptest_stdev_angle(ii) = ptest_stdev
        meth_usd(ii) = meth
        if (energyeV(ii) eq 1.0) then begin
            s1 = s1or2
            s2 = 255
        endif else begin
            s1 = 255
            s2 = s1or2
        endelse
        sbytes(0:6,ii) = [s0,s1,s2,s3,s4,s5,s6]
    endif else if (vers eq 1.37) then begin
        reads, dum, stat, meth, ctime, $
          edi6, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, r2, cl, $
          r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          amb180, ptest_mean, ptest_stdev, $
          s0,s1or2,s3,s4,s5,s6, $
          cat, caa, nat, naa, $
          format=$
          '(i3,1x,i3,1x,e13.6,1x,'+$
          '6(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),i3,1x,'+$
          '2(e13.6,1x),'+$
          '3(i3,1x),'+$
          '3(i3,1x),'+$
          '2(i3,1x),'+$
          '2(i3,1x))'
        ambig_180(ii) = amb180
        ptest_mean_angle(ii) = ptest_mean
        ptest_stdev_angle(ii) = ptest_stdev
        meth_usd(ii) = meth
        if (energyeV(ii) eq 1.0) then begin
            s1 = s1or2
            s2 = 255
        endif else begin
            s1 = 255
            s2 = s1or2
        endelse
        sbytes(0:6,ii) = [s0,s1,s2,s3,s4,s5,s6]

        nbeam_classA_towards(ii) = cat
        nbeam_classA_away(ii) = caa
        nbeam_nonA_towards(ii) = nat
        nbeam_nonA_away(ii) = naa

        if (cat ne 255) then begin
            nbeam_classA_tot(ii) = cat+caa
            nbeam_nonA_tot(ii) = nat+naa
        endif

    endif else if (vers eq 1.38) then begin

        reads, dum, stat, meth, ctime, $
          edi6, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, r2, cl, $
          r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          amb180, ptest_mean, ptest_stdev, $
          s0,s1or2,s3,s4,s5,s6, $
          cat, caa, nat, naa, $
          jsmt, e3smt0, e3smt1, e3smt2, $
          format=$
          '(i3,1x,i3,1x,e13.6,1x,'+$
          '6(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),i3,1x,'+$
          '2(e13.6,1x),'+$
          '3(i3,1x),'+$
          '3(i3,1x),'+$
          '2(i3,1x),'+$
          '2(i3,1x),'+$
          'i3,1x,3(e13.6,1x))'

        ambig_180(ii) = amb180
        ptest_mean_angle(ii) = ptest_mean
        ptest_stdev_angle(ii) = ptest_stdev
        meth_usd(ii) = meth
        if (energyeV(ii) eq 1.0) then begin
            s1 = s1or2
            s2 = 255
        endif else begin
            s1 = 255
            s2 = s1or2
        endelse
        sbytes(0:6,ii) = [s0,s1,s2,s3,s4,s5,s6]

        nbeam_classA_towards(ii) = cat
        nbeam_classA_away(ii) = caa
        nbeam_nonA_towards(ii) = nat
        nbeam_nonA_away(ii) = naa

        if (cat ne 255) then begin
            nbeam_classA_tot(ii) = cat+caa
            nbeam_nonA_tot(ii) = nat+naa
        endif
        ismt(ii) = jsmt
        edi3_smt(0:2,ii) = [e3smt0,e3smt1,e3smt2]

    endif else if (vers gt 1.38 and vers le 2.27) then begin
        
        reads, dum, stat, meth, ctime, $
          edi6, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, r2, cl, $
          r2_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          amb180, ptest_mean, ptest_stdev, $
          s0,s1or2,s3,s4,s5,s6, $
          cat, caa, nat, naa, $
          format=$
          '(i3,1x,i3,1x,e13.6,1x,'+$
          '6(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '2(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),i3,1x,'+$
          '2(e13.6,1x),'+$
          '3(i3,1x),'+$
          '3(i3,1x),'+$
          '2(i3,1x),'+$
          '2(i3,1x))'

        ambig_180(ii) = amb180
        ptest_mean_angle(ii) = ptest_mean
        ptest_stdev_angle(ii) = ptest_stdev
        meth_usd(ii) = meth
        if (energyeV(ii) eq 1.0) then begin
            s1 = s1or2
            s2 = 255
        endif else begin
            s1 = 255
            s2 = s1or2
        endelse
        sbytes(0:6,ii) = [s0,s1,s2,s3,s4,s5,s6]

        nbeam_classA_towards(ii) = cat
        nbeam_classA_away(ii) = caa
        nbeam_nonA_towards(ii) = nat
        nbeam_nonA_away(ii) = naa

        if (cat ne 255) then begin
            nbeam_classA_tot(ii) = cat+caa
            nbeam_nonA_tot(ii) = nat+naa
        endif
    endif else begin
        reads, dum, stat, meth, ctime, $
          edi6, d_scs, v_scs, e_scs, d_gse, v_gse, e_gse, r2, r2unpen, cl, $
          r2_fg, r2unpen_fg, cl_fg, rderr2, pderr2, tgerr2, $
          vgsenocorr, egsenocorr, vscgse, $
          amb180, ptest_mean, ptest_stdev, $
          s0,s1or2,s3,s4,s5,s6, $
          cat, caa, nat, naa, $
          format=$
          '(i3,1x,i3,1x,e13.6,1x,'+$
          '6(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),'+$
          '3(e13.6,1x),3(e13.6,1x),'+$
          '3(e13.6,1x),i3,1x,'+$
          '2(e13.6,1x),'+$
          '3(i3,1x),'+$
          '3(i3,1x),'+$
          '2(i3,1x),'+$
          '2(i3,1x))'

        ambig_180(ii) = amb180
        ptest_mean_angle(ii) = ptest_mean
        ptest_stdev_angle(ii) = ptest_stdev
        meth_usd(ii) = meth
        if (energyeV(ii) eq 1.0) then begin
            s1 = s1or2
            s2 = 255
        endif else begin
            s1 = 255
            s2 = s1or2
        endelse
        sbytes(0:6,ii) = [s0,s1,s2,s3,s4,s5,s6]

        nbeam_classA_towards(ii) = cat
        nbeam_classA_away(ii) = caa
        nbeam_nonA_towards(ii) = nat
        nbeam_nonA_away(ii) = naa

        if (cat ne 255) then begin
            nbeam_classA_tot(ii) = cat+caa
            nbeam_nonA_tot(ii) = nat+naa
        endif

        rchi2_unpen_cg(ii) = r2unpen
        rchi2_unpen_fg(ii) = r2unpen_fg

    endelse
    
    if (vers gt 2.02) then begin
        if (stat le 9) then status(ii) = 0 else $
          status(ii) = -1.*stat
    endif else begin
        status(ii) = stat
    endelse
    
    comptime(ii) = ctime
    rchi2_cg(ii) = r2
    clevel_cg(ii) = cl
    rchi2_fg(ii) = r2_fg
    clevel_fg(ii) = cl_fg
    
    target_mag(ii) = edi6(0)
    target_azi(ii) = edi6(2)
    error_target_mag(ii) = edi6(1)
    error2_target_mag(ii) = rderr2
    error_target_azi(ii) = edi6(3)
    error2_target_azi(ii) = pderr2
    tgyro(ii) = edi6(4)
    error_tgyro(ii) = edi6(5)
    error2_tgyro(ii) = tgerr2
    drift_step_scs(0:2,ii) = d_scs
    drift_velo_scs(0:2,ii) = v_scs
    efield_scs(0:2,ii) = e_scs
    drift_step_gse(0:2,ii) = d_gse
    drift_velo_gse(0:2,ii) = v_gse
    efield_gse(0:2,ii) = e_gse
    drift_velo_nocorr_gse(0:2,ii) = vgsenocorr
    efield_nocorr_gse(0:2,ii) = egsenocorr
    vsc_gse(0:2,ii) = vscgse

    if (vers le 1.28) then readf, unit, dum ; toaw no longer output

    readf, unit, dum
    if (nb ge nbeam_max) then message, 'Array size for beam-dimension not big enough; Must set nbeam_max keyword to be larger than : '+strtrim(nbeam_max,2)
    ddat = intarr(nb)
    reads, dum, ddat, format=ls+'(i3,1x))'
    out_beam(0:nb-1,ii) = ddat
    
    readf, unit, dum
    if (beam_load(7)) then begin
        ddat = intarr(nb)
        reads, dum, ddat, format=ls+'(i3,1x))'
        bestord_beam(0:nb-1,ii) = ddat
    endif
    
    if (vers gt 1.23 and vers le 1.28) then readf, unit, dum ; preord no longer output

    if (vers gt 1.28) then begin
        readf, unit, dum
        if (beam_load(4)) then begin
            ddat = strarr(nb)
            reads, dum, ddat, format=ls+'(a1,1x))'
            class_init_beam(0:nb-1,ii) = ddat
        endif
    endif

    if (vers gt 1.28) then begin
        readf, unit, dum
        if (beam_load(2)) then begin
            ddat = strarr(nb)
            reads, dum, ddat, format=ls+'(a1,1x))'
            class_beam(0:nb-1,ii) = ddat
        endif
    endif
    
    if (vers gt 2.27) then begin
        readf, unit, dum
        if (beam_load(3)) then begin
            ddat = strarr(nb)
            reads, dum, ddat, format=ls+'(e13.6,1x))'
            penalty_beam(0:nb-1,ii) = ddat
        endif
    endif
    
    if (vers gt 2.27) then begin
        readf, unit, dum
        ddat = strarr(nb)
        reads, dum, ddat, format=ls+'(i3,1x))'
        used_beam(0:nb-1,ii) = ddat
    endif

    record_skipped:
    
endwhile

stop_reading:

free_lun, unit
ntime = ii+1

if (ntime eq 0) then begin 
    message, 'No data within '+fname, /cont
; De-allocate space
    cfname = fname
    cenergy = energy
    cnobeam = nobeam
    cbeam_load = beam_load
    cmethod = method
    ctrange = trange
    cnostatfilter = nostatfilter
    dum = temporary(cfname)     ; Un-define the cfname flag
    dum = temporary(cenergy)    ; Un-define the cenergy flag
    dum = temporary(cnobeam)    ; Un-define the cnobeam flag
    dum = temporary(cbeam_load)
    dum = temporary(cmethod)
    dum = temporary(ctrange)
    dum = temporary(energyeV)
    dum = temporary(flip)
    dum = temporary(status)
    dum = temporary(sbytes)
    dum = temporary(nbeam_classA_tot)
    dum = temporary(nbeam_classA_towards)
    dum = temporary(nbeam_classA_away)
    dum = temporary(nbeam_nonA_tot)
    dum = temporary(nbeam_nonA_towards)
    dum = temporary(nbeam_nonA_away)
    dum = temporary(ismt)
    dum = temporary(edi3_smt)
    dum = temporary(t1)
    dum = temporary(t2)
    dum = temporary(bfield_scs)
    dum = temporary(bfield_gse)
    dum = temporary(scs2bpp_refvec)
    dum = temporary(rmax)
    dum = temporary(r_max_2)
    dum = temporary(bvar1)
    dum = temporary(bvar2)
    dum = temporary(nnn)
    dum = temporary(mm)
    dum = temporary(code_type)
    dum = temporary(nbeam_tot)
    dum = temporary(nbeam_usd)
    dum = temporary(target_mag)
    dum = temporary(target_azi)
    dum = temporary(error_target_mag)
    dum = temporary(error2_target_mag)
    dum = temporary(error_target_azi)
    dum = temporary(error2_target_azi)
    dum = temporary(tgyro)
    dum = temporary(error_tgyro)
    dum = temporary(error2_tgyro)
    dum = temporary(drift_step_scs)
    dum = temporary(drift_velo_scs)
    dum = temporary(efield_scs)
    dum = temporary(drift_step_gse)
    dum = temporary(drift_velo_gse)
    dum = temporary(efield_gse)
    dum = temporary(drift_velo_nocorr_gse)
    dum = temporary(efield_nocorr_gse)
    dum = temporary(vsc_gse)

    dum = temporary(out_beam)
    dum = temporary(used_beam)
    if (beam_load(2)) then dum = temporary(class_beam)
    if (beam_load(3)) then dum = temporary(penalty_beam)
    if (beam_load(4)) then dum = temporary(class_init_beam)
    if (beam_load(5)) then dum = temporary(maxchan_beam)
    if (beam_load(6)) then dum = temporary(qual_beam)
    if (beam_load(7)) then dum = temporary(bestord_beam)
    if (beam_load(8)) then dum = temporary(gunid_beam)
    if (beam_load(9)) then dum = temporary(xg_beam)
    if (beam_load(10)) then dum = temporary(yg_beam)
    if (beam_load(11)) then dum = temporary(alpha_beam)
    if (beam_load(12)) then dum = temporary(tof_beam)
    if (beam_load(13)) then dum = temporary(atime_ssm_beam)
    if (beam_load(14)) then dum = temporary(btime_ssm_beam)
    if (beam_load(15)) then dum = temporary(gdotb_beam)
    if (beam_load(16)) then dum = temporary(bwidth_beam)

    dum = temporary(rchi2_cg)
    dum = temporary(rchi2_unpen_cg)
    dum = temporary(clevel_cg)
    dum = temporary(rchi2_fg)
    dum = temporary(rchi2_unpen_fg)
    dum = temporary(clevel_fg)
    dum = temporary(meth_usd)
    dum = temporary(comptime)
    dum = temporary(ambig_180)
    dum = temporary(ptest_mean_angle)
    dum = temporary(ptest_stdev_angle)

    return
endif

; Define more useful time variables
date_vax = date2datevax(date)
pick_cvt_str_sec, date_vax+' 00:00:00.000000', t70mn ; midnight
t70_midnight = double(t70mn(0)) + double(t70mn(1))/1d6
t_ssm = dblarr(n_elements(t1))
twidth_sec = dblarr(n_elements(t1))
tmid = strarr(n_elements(t1))
for itime=0L,ntime-1 do begin
    pick_cvt_str_sec, t1(itime), t_start
    pick_cvt_str_sec, t2(itime), t_stop
    t_start = double(t_start(0)) + double(t_start(1))/1d6
    t_stop = double(t_stop(0)) + double(t_stop(1))/1d6
    twidth_sec(itime) = t_stop - t_start
    t_mid = (t_start + t_stop)/2.d0
    t_ssm(itime) = t_mid - t70_midnight
    t_mid = [long(t_mid),long( (t_mid-long(t_mid))*1d6 )]
    pick_cvt_sec_vax, t_mid, midstr
    tmid(itime) = midstr
endfor

; Filter the data according to energy and status and method==============
if (vers gt 1.28) then begin
    if (method eq 'ALL') then begin
        if (not nostatfilter) then $
          keep = where(energyeV eq energy and status eq 0) else $
          keep = where(energyeV eq energy)
    endif else if (method eq 'TRI') then begin
        if (not nostatfilter) then $
          keep = where(energyeV eq energy and status eq 0 and meth_usd eq 0) $
        else $
          keep = where(energyeV eq energy and meth_usd eq 0)
    endif else if (method eq 'TOF') then begin
        if (not nostatfilter) then $
          keep = where(energyeV eq energy and status eq 0 and meth_usd eq 1) $
        else $
          keep = where(energyeV eq energy and meth_usd eq 1)
    endif else if (method eq 'SMT') then begin
        if (not nostatfilter) then $
          keep = where(energyeV eq energy and status eq 0 and meth_usd eq 2) $
        else $
          keep = where(energyeV eq energy and meth_usd eq 2)
    endif else if (method eq 'RMT') then begin

;        stop

        if (not nostatfilter) then $
          keep = where(energyeV eq energy and status eq 0 and meth_usd eq 3) $
        else $
          keep = where(energyeV eq energy and meth_usd eq 3)
    endif else begin
        message, 'Value for "method" not recognized: '+method
    endelse
endif else begin
    message, 'Old file, so returning triangulation only', /cont
    if (not nostatfilter) then $
      keep = where(energyeV eq energy and status eq 1) else $
      keep = where(energyeV eq energy)
endelse

if (keep(0) eq -1) then begin
    message, 'No valid data found in this aux file', /cont
    return
endif
ntime = n_elements(keep)

t1 = t1(keep)
tmid = tmid(keep)
t2 = t2(keep)
t_ssm = t_ssm(keep)
twidth_sec = twidth_sec(keep)
bfield_scs = bfield_scs(0:2,keep)
bfield_gse = bfield_gse(0:2,keep)
scs2bpp_refvec = scs2bpp_refvec(0:2,keep)
rmax = rmax(keep)
r_max_2 = r_max_2(keep)
bvar1 = bvar1(keep)
bvar2 = bvar2(keep)
nnn = nnn(keep)
mm = mm(keep)
code_type = code_type(keep)
flip = flip(keep)

target_mag = target_mag(keep)
target_azi = target_azi(keep)
error_target_mag = error_target_mag(keep)
error_target_azi = error_target_azi(keep)
error2_target_mag = error2_target_mag(keep)
error2_target_azi = error2_target_azi(keep)
tgyro = tgyro(keep)
error_tgyro = error_tgyro(keep)
error2_tgyro = error2_tgyro(keep)
methstat = status(keep)
sbytes = sbytes(0:6,keep)

drift_step_scs = drift_step_scs(0:2,keep)
drift_velo_scs = drift_velo_scs(0:2,keep)
efield_scs = efield_scs(0:2,keep)
drift_step_gse = drift_step_gse(0:2,keep)
drift_velo_gse = drift_velo_gse(0:2,keep)
efield_gse = efield_gse(0:2,keep)
drift_velo_nocorr_gse = drift_velo_nocorr_gse(0:2,keep)
efield_nocorr_gse = efield_nocorr_gse(0:2,keep)
vsc_gse = vsc_gse(0:2,keep)
out_beam = out_beam(0:nbeam_max-1,keep)
used_beam = used_beam(0:nbeam_max-1,keep)
if (beam_load(2)) then class_beam = class_beam(0:nbeam_max-1,keep)
if (beam_load(3)) then penalty_beam = penalty_beam(0:nbeam_max-1,keep)
if (beam_load(4)) then class_init_beam = class_init_beam(0:nbeam_max-1,keep)
if (beam_load(5)) then maxchan_beam = maxchan_beam(0:nbeam_max-1,keep)
if (beam_load(6)) then qual_beam = qual_beam(0:nbeam_max-1,keep)
if (beam_load(7)) then bestord_beam = bestord_beam(0:nbeam_max-1,keep)
if (beam_load(8)) then gunid_beam = gunid_beam(0:nbeam_max-1,keep)
if (beam_load(9)) then xg_beam = xg_beam(0:nbeam_max-1,keep)
if (beam_load(10)) then yg_beam = yg_beam(0:nbeam_max-1,keep)
if (beam_load(11)) then alpha_beam = alpha_beam(0:nbeam_max-1,keep)
if (beam_load(12)) then tof_beam = tof_beam(0:nbeam_max-1,keep)
if (beam_load(13)) then atime_ssm_beam = atime_ssm_beam(0:nbeam_max-1,keep)
if (beam_load(14)) then btime_ssm_beam = btime_ssm_beam(0:nbeam_max-1,keep)
if (beam_load(15)) then gdotb_beam = gdotb_beam(0:nbeam_max-1,keep)
if (beam_load(16)) then bwidth_beam = bwidth_beam(0:nbeam_max-1,keep)

ismt = ismt(keep)
edi3_smt = edi3_smt(0:2,keep)

nbeam_tot = nbeam_tot(keep)
nbeam_classA_tot = nbeam_classA_tot(keep)
nbeam_classA_towards = nbeam_classA_towards(keep)
nbeam_classA_away = nbeam_classA_away(keep)
nbeam_nonA_tot = nbeam_nonA_tot(keep)
nbeam_nonA_towards = nbeam_nonA_towards(keep)
nbeam_nonA_away = nbeam_nonA_away(keep)

; Define nbeam_usd
nbeam_usd = make_array(size=size(nbeam_tot), value=0)
if (vers le 1.28) then goodstat = 1 else goodstat = 0
for ii=0L,ntime-1 do begin
    
    if (vers le 2.27) then begin
        if (methstat(ii) eq goodstat) then begin        
            id = where(out_beam(0:nbeam_tot(ii)-1,ii) le 1)
            if (id(0) ne -1) then nbeam_usd(ii) = n_elements(id)
        endif
        
        if (nbeam_classA_tot(ii) ne 255) then begin
            if (nbeam_classA_tot(ii)+nbeam_nonA_tot(ii) ne nbeam_usd(ii)) then $
              message, 'Something wrong with counting statistics'
        endif
    endif else begin
        if (methstat(ii) eq goodstat) then begin        
            id = where(used_beam(0:nbeam_tot(ii)-1,ii) eq 1)
            if (id(0) ne -1) then nbeam_usd(ii) = n_elements(id)
        endif
    endelse
    
endfor

rchi2_cg = rchi2_cg(keep)
rchi2_unpen_cg = rchi2_unpen_cg(keep)
clevel_cg = clevel_cg(keep)
rchi2_fg = rchi2_fg(keep)
rchi2_unpen_fg = rchi2_unpen_fg(keep)
clevel_fg = clevel_fg(keep)

comptime = comptime(keep)
meth_usd = meth_usd(keep)
ambig_180 = ambig_180(keep)
ptest_mean_angle = ptest_mean_angle(keep)
ptest_stdev_angle = ptest_stdev_angle(keep)

cfname = fname
cenergy = energy
cnobeam = nobeam
cbeam_load = beam_load
cmethod = method
ctrange = trange
cnostatfilter = nostatfilter
already_in_buffer:
;==============================================================
;==============================================================
fname_out = cfname
ntime_out = ntime
methstat_out = methstat
sbytes_out = sbytes
t1_out = t1
tmid_out = tmid
t2_out = t2
t_ssm_out = t_ssm
twidth_sec_out = twidth_sec
bfield_scs_out = bfield_scs
bfield_gse_out = bfield_gse
scs2bpp_refvec_out = scs2bpp_refvec
rmax_out = rmax
r_max_2_out = r_max_2
bvar1_out = bvar1
bvar2_out = bvar2
nnn_out = nnn
mm_out = mm
code_type_out = code_type

nbeam_usd_out = nbeam_usd
nbeam_classA_tot_out = nbeam_classA_tot
nbeam_classA_towards_out = nbeam_classA_towards
nbeam_classA_away_out = nbeam_classA_away
nbeam_nonA_tot_out = nbeam_nonA_tot
nbeam_nonA_towards_out = nbeam_nonA_towards
nbeam_nonA_away_out = nbeam_nonA_away
nbeam_tot_out = nbeam_tot

ismt_out = ismt
edi3_smt_out = edi3_smt

flip_out = flip

target_mag_out = target_mag
target_azi_out = target_azi
error_target_mag_out = error_target_mag
error_target_azi_out = error_target_azi
error2_target_mag_out = error2_target_mag
error2_target_azi_out = error2_target_azi
tgyro_out = tgyro
error_tgyro_out = error_tgyro
error2_tgyro_out = error2_tgyro

drift_step_scs_out = drift_step_scs
drift_velo_scs_out = drift_velo_scs
efield_scs_out = efield_scs
drift_step_gse_out = drift_step_gse
drift_velo_gse_out = drift_velo_gse
efield_gse_out = efield_gse
drift_velo_nocorr_gse_out = drift_velo_nocorr_gse
efield_nocorr_gse_out = efield_nocorr_gse
vsc_gse_out = vsc_gse

out_beam_out = out_beam
used_beam_out = used_beam
if (beam_load(2)) then class_beam_out = class_beam
if (beam_load(3)) then penalty_beam_out = penalty_beam
if (beam_load(4)) then class_init_beam_out = class_init_beam
if (beam_load(5)) then maxchan_beam_out = maxchan_beam
if (beam_load(6)) then qual_beam_out = qual_beam
if (beam_load(7)) then bestord_beam_out = bestord_beam
if (beam_load(8)) then gunid_beam_out = gunid_beam
if (beam_load(9)) then xg_beam_out = xg_beam
if (beam_load(10)) then yg_beam_out = yg_beam
if (beam_load(11)) then alpha_beam_out = alpha_beam
if (beam_load(12)) then tof_beam_out = tof_beam
if (beam_load(13)) then atime_ssm_beam_out = atime_ssm_beam
if (beam_load(14)) then btime_ssm_beam_out = btime_ssm_beam
if (beam_load(15)) then gdotb_beam_out = gdotb_beam
if (beam_load(16)) then bwidth_beam_out = bwidth_beam

rchi2_cg_out = rchi2_cg
rchi2_unpen_cg_out = rchi2_unpen_cg
clevel_cg_out = clevel_cg
rchi2_fg_out = rchi2_fg
rchi2_unpen_fg_out = rchi2_unpen_fg
clevel_fg_out = clevel_fg
meth_usd_out = meth_usd
comptime_out = comptime
ambig_180_out = ambig_180
ptest_mean_angle_out = ptest_mean_angle
ptest_stdev_angle_out = ptest_stdev_angle

if (keyword_set(rotmat)) then begin
    
    if (vers le 1.23) then begin
        message, 'bpp2scs rotation matrix not available because there was an error in the EDI_PISO code regarding this rotation', /cont
        bpp2scs_out = fltarr(3,3,ntime)+rfill
    endif else if (vers eq 1.24) then begin
        bpp2scs_out = fltarr(3,3,ntime)
        for i=0L,ntime-1 do begin
            bhat_scs = bfield_scs(*,i)/sqrt(total(bfield_scs(*,i)^2))
            if flip(i) then factor = -1.d0 else factor = 1.d0
            z = factor*bhat_scs
            r = [0.,factor,0.]
            x = [r(1)*z(2)-r(2)*z(1), $ ; r cross z
                 r(2)*z(0)-r(0)*z(2), $
                 r(0)*z(1)-r(1)*z(0)]
            x = x/sqrt(total(x^2)) ; Normalized to unit length
            y = [z(1)*x(2)-z(2)*x(1), $ ; z cross x
                 z(2)*x(0)-z(0)*x(2), $
                 z(0)*x(1)-z(1)*x(0)]
            bpp2scs_out(*,2,i) = z
            bpp2scs_out(*,0,i) = x
            bpp2scs_out(*,1,i) = y
        endfor
    endif else begin
        bpp2scs_out = fltarr(3,3,ntime)
        for i=0L,ntime-1 do begin
            bhat_scs = bfield_scs(*,i)/sqrt(total(bfield_scs(*,i)^2))
            if flip(i) then factor = -1.d0 else factor = 1.d0
            z = factor*bhat_scs
            r = scs2bpp_refvec(*,i)
            x = [r(1)*z(2)-r(2)*z(1), $ ; r cross z
                 r(2)*z(0)-r(0)*z(2), $
                 r(0)*z(1)-r(1)*z(0)]
            x = x/sqrt(total(x^2)) ; Normalized to unit length
            y = [z(1)*x(2)-z(2)*x(1), $ ; z cross x
                 z(2)*x(0)-z(0)*x(2), $
                 z(0)*x(1)-z(1)*x(0)]
            bpp2scs_out(*,2,i) = z
            bpp2scs_out(*,0,i) = x
            bpp2scs_out(*,1,i) = y
        endfor
    endelse
endif

rfill_out = rfill

stat_out = 1

return
end
