;	edi_piso.pro,v 1.54 2004/06/29 22:12:01 ppq Exp	
pro edi_piso, log_unit
;+
; ==========================================================================
; edi_piso = Electron Drift Instrument Principal Investigator Software,
;            an IDL program for rapid data analysis of the EDI instrument
;            of the satellite mission CLUSTER-II;
;            spin-averaged data (in the form of an IFF file)
;            for the CSDS are generated
; ==========================================================================
; Program development based on the EDIQ software packet and EQ-S 
; experiences of NOS, the 'pseudocode.txt' of GEP which at that 
; time based on 'tspl06.pro' of HAV including some gyro projection 
; subroutines and the 'bestarg.pro' packet of THB.
;
; History:
; ========
; August 2000 - First running version (EQ-S data verification)
;               Curator: Matthias Foerster, mfo@mpe.mpg.de
; 07 Sep 2000 - New magnetic field data file layout in agreement w/ Edita (EG)
; 12 Sep 2000 - Incorporation of the 'whatech.pro' and 'bestarg.pro'
;               subroutines
; 11 Okt 2000 - Renamed 'edi_piso.pro'
;               New curator: Pamela Puhl-Quinn, ppq@mpe.mpg.de
;               Many changes made to accomodate analysis of Cluster/EDI data
;               See snapshot_history to see further history.
;-

; CPU time/Memory calls
systime_start=systime(/seconds)
start_mem = MEMORY(/CURRENT)

; This common block has been defined already because
; edi_piso_wrapper.pro calls ep_envar_con.pro first.
common ep_envar_con_cb

common iff_hourly_cb, iff_output

;=======================================================
;             DATA FILE INFO/OPENING
;=======================================================
status = edi_piso_file_info(finfo,sc_number=sc_number)
log_unit = finfo.log_unit

if (not status) then begin
    printf, finfo.log_unit, 'File information extraction not successful'

    openw, iff_unit, finfo.iff_name, /get_lun
    printf, iff_unit, '%M '+finfo.iff_name+' opened at '+systime()
    edi_piso_write_iff_header, iff_unit, finfo, sc_number
    printf, iff_unit, '%M '+'File information extraction not successful'

; Close iff file
; But first, write a fill record, the 'E' line, and a timing message
    edi_piso_write_iff_data, 0, iff_unit, 0., 0., /fill
    printf, iff_unit, 'E 1'
    printf, iff_unit, '%M '+finfo.iff_name+' closed at '+systime()
    free_lun, iff_unit
    
    return
endif

; Safety check on pp_qual_min=========================================
; pp_qual_min = 0 is not allowed - This is a safety because originally
;                                in whatech.pro it was assumed that
;                                this is OK, but it's not, so we make
;                                sure here that these points aren't
;                                ever used.
if (pp_qual_min eq 0) then begin
    printf, finfo.log_unit, 'pp_qual_min=0 is not allowed;  Beams with qual=0 do not have angle track, let alone time track'
    return
endif

printf, finfo.log_unit, 'RUN BEGIN TIME: '+systime()

openw, iff_unit, finfo.iff_name, /get_lun
printf, iff_unit, '%M '+finfo.iff_name+' opened at '+systime()
edi_piso_write_iff_header, iff_unit, finfo, sc_number

openw, pstat_unit, finfo.pstat_name, /get_lun

if (pp_write_aux) then begin
    openw,aux_unit,finfo.aux_name,/get_lun
    edi_piso_write_aux_header, aux_unit, finfo, sc_number
endif

if (pp_write_cdf) then begin
    ep_create_edi_archive_cdf, sc_number, finfo
    cdf_unit = cdf_open(finfo.cdf_name+'.cdf')
endif

if (pp_ediefw_out) then begin
    openw, ediefw_unit, finfo.ediefw_name, /get_lun
    printf, ediefw_unit, '%M '+finfo.ediefw_name+' opened at '+systime()
    printf, ediefw_unit, '%M SPECIAL EDI/EFW COMPARISON IFFALL FILE!!!!'
    printf, ediefw_unit, '%M E-DRIFT IS IN SCS, EARTHFRAME HERE (not satellite frame)!!!!'
    printf, ediefw_unit, '%M V-DRIFT NOT WRITTEN OUT (FILLS ONLY)!!!!'
    printf, ediefw_unit, '%M OTHERWISE, SAME STRUCTURE AS NORMAL IFFALL FILE'
    edi_piso_write_iff_header, ediefw_unit, finfo, sc_number
endif

if (pp_edidsi_out) then begin
    openw, edidsi_unit, finfo.edidsi_name, /get_lun
    printf, edidsi_unit, '%M '+finfo.edidsi_name+' opened at '+systime()
    printf, edidsi_unit, '%M SPECIAL EDI/EFW COMPARISON FILE!'
    printf, edidsi_unit, '%M E-DRIFT IS IN DSI (Despun Satellite Inverted), SATELLITE FRAME (no correction for satellite motion)!'
    printf, edidsi_unit, '%M V-DRIFT NOT WRITTEN OUT (FILLS ONLY)!'
    printf, edidsi_unit, '%M OTHERWISE, SAME STRUCTURE AS NORMAL IFFALL FILE'
    edi_piso_write_iff_header, edidsi_unit, finfo, sc_number
endif

if (pp_edidsi_inertial_out) then begin
    openw, edidsi_inertial_unit, finfo.edidsi_inertial_name, /get_lun
    printf, edidsi_inertial_unit, '%M '+finfo.edidsi_inertial_name+' opened at '+systime()
    printf, edidsi_inertial_unit, '%M SPECIAL EDI/EFW COMPARISON FILE!'
    printf, edidsi_inertial_unit, '%M E-DRIFT IS IN DSI (Despun Satellite Inverted), INERTIAL FRAME (Correction for satellite motion made)!'
    printf, edidsi_inertial_unit, '%M V-DRIFT NOT WRITTEN OUT (FILLS ONLY)!'
    printf, edidsi_inertial_unit, '%M OTHERWISE, SAME STRUCTURE AS NORMAL IFFALL FILE'
    edi_piso_write_iff_header, edidsi_inertial_unit, finfo, sc_number
endif

write_diag_file = 0
if (write_diag_file) then begin
    openw, diag_unit, 'runner_diag_meth'+string(pp_runner_penalty_meth,'(i1)')+'.txt', /get_lun
endif

;=======================================================
;          DATA EXTRACTION AND ANALYSIS
;=======================================================
@hk_struct
@pick_const
@pick_struct

; Analyze the spin according to 'pp_nchunk' assignment:
;     pp_nchunk = 1      whole spin, 4s resolution
;     pp_nchunk = 2      2s resolution
;     pp_nchunk = 4      1s resolution
if (pp_nchunk eq 1) then use_hrfgm = 0 else use_hrfgm = 1

; Decide which hours to process based on Hans' qstat
; qstat.hourstat(ihour) = 0 means don't process this hour
; qstat.hourstat(ihour) = 1 means process this hour where
; ihour runs from 0 to 23
qstat = ep_qstat_read(finfo.date, sc_number)
if (not qstat.status) then begin
    printf, finfo.log_unit, qstat.msg_out
    printf, iff_unit, '%M '+qstat.msg_out

; Close iff file
; But first, write a fill record, the 'E' line, and a timing message
    edi_piso_write_iff_data, 0, iff_unit, 0., 0., /fill
    printf, iff_unit, 'E 1'
    printf, iff_unit, '%M '+finfo.iff_name+' closed at '+systime()
    free_lun, iff_unit
    
    return
endif

; Hour-by-hour data extraction based on starting at midnight
midnight_str = finfo.date_vax+' 00:00:00.000'
pick_cvt_str_sec, midnight_str, t70_mn
t70_midnight = double(t70_mn(0)) + double(t70_mn(1))/1d6 ; decimal t70
t70_midnight0 = t70_mn(0)
t70_midnight1 = t70_mn(1)

; Create and read the Spin-resolution FGM file for this day
;spawn, ['ls',pp_edi_piso_att_orb_path], s, /noshell
;print, 'ATT_ORB Pre-spin: '
;print, s
spinfgm_stat = ep_get_spinfgm(finfo.date, sc_number, $
                              pp_edi_piso_fgm_out, pp_edi_piso_fgm_save, $
                              pp_gdc_rdda_root, $
                              pp_gdc_rdda_root2, $
                              pp_edi_piso_fgm_root, $
                              pp_edi_piso_att_orb_path, $
                              pp_edi_piso_bash, $
                              finfo.log_unit, iff_unit, $
                              pp_ep_code_path, $
                              msg_out, pp_spinfgm_file, ddata)
if (not spinfgm_stat) then begin
    printf, finfo.log_unit, msg_out
    printf, iff_unit, '%M '+msg_out
    
; Close iff file
; But first, write a fill record, the 'E' line, and a timing message
    edi_piso_write_iff_data, 0, iff_unit, 0., 0., /fill
    printf, iff_unit, 'E 1'
    printf, iff_unit, '%M '+finfo.iff_name+' closed at '+systime()
    free_lun, iff_unit

    return
endif

; Create an array of indices , inds, based on spin-avged mag. field times.
; Since these times have a resolution of 4 seconds, indices created
; from integerizing them (after converting them to
; seconds-since-midnight) are unique and manageable.

t70_fgm = reform(ddata(0,*))
inds = long(t70_fgm - t70_midnight)

; Construct the time tag for the field data in seconds since midnight, ssm
dum = transpose( [ [long(t70_fgm)],[long( (t70_fgm - long(t70_fgm))*1d6 )] ] )
dum = sub_ctime( dum, [t70_midnight0,t70_midnight1] ) 

; Place the time and field information on a grid at these indices
bt_grid_ssm = make_array(86400,/float,value=pp_rfill)
bt_grid_t70 = make_array(86400,/double,value=pp_rfill)
bx_grid = make_array(86400,/float,value=pp_rfill)
by_grid = make_array(86400,/float,value=pp_rfill)
bz_grid = make_array(86400,/float,value=pp_rfill)
bvar1_grid = make_array(86400,/float,value=pp_rfill)
bvar2_grid = make_array(86400,/float,value=pp_rfill)
bt_grid_ssm(inds) = reform(double(dum(0,*)) + double(dum(1,*))/1d6) ; ssm
bt_grid_t70(inds) = t70_fgm     ; t70
bx_grid(inds) = reform(ddata(1,*)) ; nT
by_grid(inds) = reform(ddata(2,*)) ; nT
bz_grid(inds) = reform(ddata(3,*)) ; nT
bvar1_grid(inds) = reform(ddata(4,*))
bvar2_grid(inds) = reform(ddata(5,*))
ddata = 0

; Spin-by-spin common block ('os' = one spin)
common edi_piso_onespin_common, spin_start_ssm_os, spin_width_sec_os, spin_srt_ssm_os

; Chunk-by-chunk common block ('oc' = one chunk)
common edi_piso_onechunk_common, bx_oc, by_oc, bz_oc, atime_ssm_oc, btime1_ssm_oc, btime2_ssm_oc, t1_oc, t1m2_oc, VAX1_oc, VAY1_oc, VAX2_oc, VAY2_oc, I1_oc, I2_oc, SQ1_oc, SQ2_oc, NNN_oc, mm_oc, code_type_oc, mxch1_oc, mxch2_oc

; Hour-by-hour common block
common edi_piso_onehour_common, btime1,btime2,atime0,atime1,t1,t1m2,VAX1,VAY1,VAX2,VAY2,I1,I2,SQ1,SQ2,E_INDEX,NNN,mm,code_type,sob,limhi,limlo,cnum,inmodstr,mxch1,mxch2

; Initialize the CDF record number counter
recnum = 0L
recnum_ediefw = 0L
recnum_edidsi = 0L
recnum_edidsi_inertial = 0L


; Initialize the Inst_mode string
inmod_old = 'EDI_??_??_??_??_??_??_??_??'

; Get the start and stop times as specified by the user.
tinfo = getenv('EDI_PISO_TINFO')
if (tinfo eq 'ALL') then begin
    begt = 0d0
    endt = 23d0*3600d0+59d0*60d0+59.9d0 ; ssm
endif else begin
    begt = double(strmid(tinfo,0,2))*3600d0 + $
      double(strmid(tinfo,3,2))*60d0 + $
      double(strmid(tinfo,6,2))
    endt = double(strmid(tinfo,9,2))*3600d0 + $
      double(strmid(tinfo,12,2))*60d0 + $
      double(strmid(tinfo,15,2)) - .1 ; Fudge for the hour-boundary case
endelse

ihr1 = long(begt/3600d0)
ihr2 = long(endt/3600d0)
hrstat = intarr(24)
hrstat(ihr1:ihr2) = 1

;==========================================================
; PSTAT Init
;==========================================================
; NOTE: Failure-bookkeeping (nspin_edata_fail_stats) done using the
; reason-structure already defined by pp_pstat (see ep_envar_con.pro)

pstat = $                       ; SPIN stats:
  {nspin_total:lonarr(24), $    ; # of spins in the entire hour
   $                            ; as determined by looping over all 
   $                            ; FGM Spinres data in this hour
   $
   nspin_user:lonarr(24), $     ; # of spins that fall within the 
                                ; time interval desired by the user
   $
   nspin_edata:lonarr(24,2), $ ; # of spins with enough data for anal.
   $                             ; DIM=[ihour,ienergy]
   nspin_edata_succ:lonarr(24,2), $ ; Success, Output to IFF/AUX files
   nspin_edata_succ_good:lonarr(24,2), $
   nspin_edata_succ_bad:lonarr(24,2), $
   nspin_edata_succ_caut:lonarr(24,2), $
   nspin_edata_succ_tri:lonarr(24,2), $
   nspin_edata_succ_pmt:lonarr(24,2), $
   nspin_edata_succ_smt:lonarr(24,2), $
   nspin_edata_succ_rmt:lonarr(24,2), $
   nspin_edata_fail:lonarr(24,2), $ ; Fail, No output
   nspin_edata_fail_stats:lonarr(24,2,n_elements(pp_pstat)), $
   $                            ; BEAM stats (only for the successful spins)
   nbeam_edata_succ:lonarr(24,2), $
   nbeam_edata_succ_classA:lonarr(24,2), $
   nbeam_edata_succ_classB:lonarr(24,2), $
   nbeam_edata_succ_classC:lonarr(24,2), $
   nbeam_edata_succ_classD:lonarr(24,2), $
   nbeam_edata_succ_classG:lonarr(24,2), $     
   nbeam_edata_succ_classR:lonarr(24,2), $
   nbeam_edata_succ_trioutlier:lonarr(24,2)}

for ihour=0,23 do begin         ; Process hour by hour
                                ; Whole day of BSD data would be too
                                ; much to store all at once
; Jump ahead if necessary
    if hrstat(ihour) then begin
        printf, finfo.log_unit, 'Hour desired as specified by user: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        printf, iff_unit, '%M '+'Hour desired as specified by user: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
    endif else begin
        printf, finfo.log_unit, 'Hour not desired as specified by user: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        printf, iff_unit, '%M '+'Hour not desired as specified by user: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        goto, next_hour
    endelse

    if (qstat.hourstat(ihour)) then begin
        printf, finfo.log_unit, 'Hour deemed worthy by qstat: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        printf, iff_unit, '%M '+'Hour deemed worthy by qstat: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
    endif else begin
        printf, finfo.log_unit, 'Hour deemed unworthy by qstat: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        printf, iff_unit, '%M '+'Hour deemed unworthy by qstat: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        goto, next_hour
    endelse
    
; Check to see if we have any spin-averaged magnetic field data for
; this hour.
    ind1 = long(86400./24.*ihour)
    ind2 = long(86400./24.*(ihour+1))-1
    
    bt_ssm = bt_grid_ssm(ind1:ind2)
    bt_t70 = bt_grid_t70(ind1:ind2)
    ib = where(bt_ssm ne pp_rfill)
    if (ib(0) eq -1) then begin
        printf, finfo.log_unit, 'No magnetic field data for this hour: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        printf, iff_unit, '%M '+'No magnetic field data for this hour: ' + $
          string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        goto, next_hour
    endif
    printf, finfo.log_unit, 'Magnetic field data exists for this hour: ' + $
      string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
    printf, iff_unit, '%M '+'Magnetic field data exists for this hour: ' + $
      string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'    
    bt_ssm = bt_ssm(ib)
    bt_t70 = bt_t70(ib)
    bx = bx_grid(ind1:ind2) & bx = bx(ib)
    by = by_grid(ind1:ind2) & by = by(ib)
    bz = bz_grid(ind1:ind2) & bz = bz(ib)
    bvar1 = bvar1_grid(ind1:ind2) & bvar1 = bvar1(ib)
    bvar2 = bvar2_grid(ind1:ind2) & bvar2 = bvar2(ib)

; Initialize the variables that will be augmented packet by
; packet and then will have the first element cropped off.    
    srt0         = 0L           ; Sun Reference Time (sec since 1970)
    srt1         = 0L           ;  "               " (microsec)
    tspin        = 0L           ; spin period (microsec)
    
    
; Initialize the variables that will be augmented subpacket by
; subpacket and then will have the first element cropped off.
    ep_augment_oh_data, /init
    
; Define the time range for this hour (which allows for buffers on
; each end to allow for those spin intervals which cross hour boundaries)
    hbl = t70_midnight + double(ihour)*3600.d0 ; Hour boundary left, dec. T70
    hbr = hbl + 3600.d0         ; Hour boundary right, dec. T70
    
    t = hbl-pp_hour_delta
    at0 = long(t)
    at1 = long(double(t-long(t))*1d6)
    pick_cvt_sec_vax, [at0,at1], tstr_start
    
    t = hbr+pp_hour_delta
    at0 = long(t)
    at1 = long(double(t-long(t))*1d6)
    pick_cvt_sec_vax, [at0,at1], tstr_end

; Get 1st packet for this hour
    pick_init, finfo.suf_name, finfo.raw_name
    if n_elements(sn) ne 0 then dum = temporary(sn)
    sn = pickst_sc(pid,tstr_start,status) ; pid not used on input

    printf, finfo.log_unit, 'Initial pickst_sc pid/status for this hour: ', $
      pid,status
    printf, iff_unit, '%M '+'Initial pickst_sc pid/status for this hour: ', $
      pid,status

    if (status) then begin
        tpack = double(sn.hdr.packet_time(0)) + $
          double(sn.hdr.packet_time(1))/1d6
        
        printf, finfo.log_unit, 'tstr_start defined for this hour: ', $
          tstr_start
        printf, iff_unit, '%M '+'tstr_start defined for this hour: ', $
          tstr_start
        
        pick_cvt_sec_vax, sn.hdr.packet_time, tpackstr
        printf, finfo.log_unit, '1st tpack returned for this hour: ', $
          tpackstr
        printf, iff_unit, '%M '+'1st tpack returned for this hour: ', $
          tpackstr
        
        printf, finfo.log_unit, 'tstr_end defined for this hour  : ', $
          tstr_end
        printf, iff_unit, '%M '+'tstr_end defined for this hour  : ', $
          tstr_end
    endif else goto, hour_read_out
    
;ToF-Diag    openw, 55, 'ToF_diag_h'+strtrim(ihour,2)+'.txt'
    
    while tpack lt hbr+pp_hour_delta do begin
        
; Check that the operational modes of EDI and the S/C are what we
; desire.
        checknum = 0
        if (not ep_opmode_check(sn,checknum,mode_fields)) then begin
            ep_augment_oh_data, sn, checknum, mode_fields
;            pick_cvt_sec_vax, sn.hdr.packet_time, tpackstr
;            printf, finfo.log_unit, tpackstr+' '+pp_checkmsg_array(checknum)
            goto, next_packet
        endif
        
; At this point in the code, we definitely have science mode 5,
; non-ram dump data and don't have to worry that 'sn' doesn't have
; some of the fields required for the next check.
; Check certain things related to nnn/code-type/sample-type violations
; and telemetry and/or ground software errors
        if (not ep_errorfree_check(sn,checknum,mode_fields)) then begin
            ep_augment_oh_data, sn, checknum, mode_fields
;            pick_cvt_sec_vax, sn.hdr.packet_time, tpackstr
;            printf, finfo.log_unit, tpackstr+' '+pp_checkmsg_array(checknum)
            goto, next_packet
        endif
        
;=============================================================
;=============================================================
;EXTRACT THIS PACKET -EXTRACT THIS PACKET -EXTRACT THIS PACKET 
;NOW THAT YOU'VE PASSED THE ABOVE CHECKS!
        
; srt and tspin have a different resolution than the EDI subpacket data
        srt0 = [srt0,sn.srt(0)]
        srt1 = [srt1,sn.srt(1)]
        tspin = [tspin,sn.tspin]
        if (sn.srt1(0) ne 0) then begin
            srt0 = [srt0,sn.srt1(0)]
            srt1 = [srt1,sn.srt1(1)]
            dum = sub_ctime(sn.srt1,sn.srt)
            dum = long( double(dum(0))*1d6 + double(dum(1)) )
            tspin = [tspin,dum]
        endif

; Augment the subpacket data
        ep_augment_oh_data, sn, checknum, mode_fields


; DIAGS:
;        print, n_elements(srt0)
;        print, n_elements(srt1)
;        print, n_elements(tspin)
;        print, n_elements(atime0)
;        print, n_elements(atime1)
;        print, n_elements(btime1)
;        print, n_elements(btime2)
;        stop

;=============================================================        
;=============================================================
        
; Get next packet
        next_packet:
        dum = temporary(sn)
        sn = pickst_sc(pid,tstr_start,status)
        if (status) then $
          tpack = double(sn.hdr.packet_time(0)) + $
          double(sn.hdr.packet_time(1))/1d6 else goto, hour_read_out
        
    endwhile

;ToF-Diag    close, 55
;ToF-Diag    stop

    hour_read_out:pick_close

    data_ok = 0
    ndata = n_elements(atime0)  ; EDI data samples
    if (ndata gt 1) then begin
        
        ep_augment_oh_data, /crop
        ndata = ndata - 1

; atime in seconds since midnight
        atime = transpose( [ [atime0], [atime1] ] )
        atime_ssm = sub_ctime(atime,[t70_midnight0,t70_midnight1])
        atime_ssm = reform(double(atime_ssm(0,*)) + double(atime_ssm(1,*))/1d6)
        atime_t70 = double(atime0) + double(atime1)/1d6
        
; Check the mode information which is within this hour.
; Could have the case here where we have only records
; indicating either ep_opmode_check failed, or
; ep_errorfree_check failed, and placeholders were made
; so that this info gets out to the IFF file.
        
        cnumeq0 = where(cnum eq 0)
        if (cnumeq0(0) eq -1) then begin ; No valid EDI data
            
; Output to the .IFF file 'Instr_mode' and fill data lines.
; This has the same form as the check below, that's why it's a
; little clunky here...
            id0 = lindgen(ndata)
            bad = where(cnum(id0) ne 0)
            if (bad(0) ne -1) then begin
                for ibad=0,n_elements(bad)-1 do begin
                    inmod = inmodstr(id0(bad(ibad)))
                    chnum = cnum(id0(bad(ibad)))
                    otime70 = [atime0(id0(bad(ibad))),atime1(id0(bad(ibad)))]
                    otimessm = atime_ssm(id0(bad(ibad)))
                    if (chnum le 3 or chnum eq 10) then begin
                        if (inmod ne inmod_old) then begin
                            ep_write_iff_instmode, recnum, iff_unit, $
                              sc_number, inmod, pp_checkmsg_array(chnum), $
                              otime70
                            edi_piso_write_iff_data, recnum, iff_unit, $
                              otimessm, rectime_old_ssm, /fill
                            recnum = recnum + 1L
                            inmod_old = inmod
                        endif
                    endif else begin
                        ep_write_iff_checkmsg, iff_unit, inmod, $
                          pp_checkmsg_array(chnum), otime70
                    endelse
                endfor
            endif

	    isort = sort(cnum)
            cnum_s = cnum(isort)
            iuniq = uniq(cnum_s)
            emsg = 'All EDI data within this hour failed opmode/error_free checks :'
            printf, iff_unit, '%M ==========================================='
            printf, iff_unit, '%M '+ emsg
            printf, iff_unit, '%M '+string(ihour,'(i2.2)')+':00 - '+ $
                string(ihour+1,'(i2.2)')+':00'
            printf, iff_unit, '%M Failure messages found:'

            printf, finfo.log_unit, '========================================'
            printf, finfo.log_unit, emsg
            printf, finfo.log_unit, string(ihour,'(i2.2)')+':00 - '+ $
                string(ihour+1,'(i2.2)')+':00'
            printf, finfo.log_unit, 'Failure messages found:'

            for i=0,n_elements(iuniq)-1 do begin
	        printf, iff_unit, '%M    '+pp_checkmsg_array(cnum_s(iuniq(i)))
	        printf, finfo.log_unit, '    '+pp_checkmsg_array(cnum_s(iuniq(i)))
            endfor
            printf, iff_unit, '%M ==========================================='
            printf, finfo.log_unit, '========================================'

            goto, next_hour

        endif else data_ok = 1

    endif

    shk_ok = 0
    nshk = n_elements(srt0)     ; SHK samples
    if (nshk gt 1) then begin
        shk_ok = 1

; Crop off the 1st element
        tspin     = tspin(1:nshk-1)
        srt0      = srt0(1:nshk-1)
        srt1      = srt1(1:nshk-1)
        nshk = nshk - 1
        
; srt in  seconds since midnight
        srt = transpose( [ [srt0], [srt1] ] )
        srt_ssm = sub_ctime( srt, [t70_midnight0,t70_midnight1] )
        srt_ssm = reform(double(srt_ssm(0,*)) + double(srt_ssm(1,*))/1d6)
        srt_t70 = double(srt0) + double(srt1)/1d6
    endif

    if (not shk_ok or not data_ok) then begin
        if (not data_ok) then begin
            printf, iff_unit, '%M '+ $
              'No EDI data for this hour: ' + $
              string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
            printf, finfo.log_unit, 'No EDI data for this hour: ' + $
              string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        endif
        if (not shk_ok) then begin
            printf, iff_unit,  '%M '+ $
              'No SHK data for this hour: ' + $
              string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
            printf, finfo.log_unit, 'No SHK data for this hour: ' + $
              string(ihour,'(i2.2)')+':00 - '+string(ihour+1,'(i2.2)')+':00'
        endif
        goto, next_hour
    endif

;=============================================================
; ANALYZE THIS HOUR OF DATA - ANALYZE THIS HOUR OF DATA - 
;=============================================================

; Get the high-res bfield for this hour
    if (use_hrfgm or pp_order_use_hrb) then begin
;        spawn, ['ls',pp_edi_piso_att_orb_path], s, /noshell
;        print, 'ATT_ORB Pre-HR: '
;        print, s
        hrfgm_stat = ep_get_hrfgm_oh(finfo.date, sc_number, ihour, $
                                     t70_midnight0, t70_midnight1, $
                                     pp_edi_piso_fgm_out, $
                                     pp_edi_piso_fgm_save, $
                                     pp_gdc_rdda_root, $
                                     pp_gdc_rdda_root2, $
                                     pp_edi_piso_fgm_root, $
                                     pp_edi_piso_att_orb_path, $
                                     pp_edi_piso_bash, $
                                     finfo.log_unit, iff_unit, $
                                     pp_ep_code_path, $
                                     msg_out, pp_hrfgm_file, $
                                     hrbt_ssm, hrbv_scs)

        if (not hrfgm_stat) then begin
            printf, finfo.log_unit, $
              'High-res FGM data production unsuccessful for this hour'
            printf, finfo.log_unit, $
              'Skipping to next hour'
            printf, iff_unit, '%M '+ $
              'High-res FGM data production unsuccessful for this hour'
            printf, iff_unit, '%M '+ $
              'Skipping to next hour'
            goto, next_hour
        endif
    endif else hrfgm_stat = 0

    ntot = n_elements(btime1)    
; Send this hour of information to Hans Vaith's 'runest.pro' in order
; to make definitive order assignments (or at least give initial
; guesses).
;    tof =                       ; time of flight in micro-s
;    st =                        ; beamtime in seconds since 1.1.1970
;    ctype =                     ; correlator code type (0=short)
;    nc =                        ; correlator n (1,2,4,8,16,32,64)
;    mc =                        ; correlator m (16,8,4,2 )
;    gdu =                       ; Gun/Det-Unit #
;    qual =                      ; Beam quality (1,2, or 3)
;    bb_sr =                     ; spin-averaged Bmag data
;    st_sr =                     ; associated times (seconds since 1.1.1970)
;    maxorder =                  ; Maximum order considered
    ep_prep_runest, tof, st, ctype, nc, mc, gdu, qual, id, id1, id2, prepstat
    if (prepstat) then begin

        if (pp_order_use_hrb and hrfgm_stat) then begin

;            save, pp_maxorder_runest, tof, st, ctype, nc, mc, gdu, qual, $
;              hrbv_scs, hrbt_ssm, t70_midnight, pp_runest_smooth, $
;              file='011009_c1_13-14_runest_in.idlsav'

            if (pp_runest_sfac eq 0) then $
              havout = RunEst(maxorder=pp_maxorder_runest, $
                              tof=tof, stime=st, ctype=ctype, nc=nc, $
                              mc=mc, gdu=gdu, q=qual, $
                              bb_sr=sqrt(total(hrbv_scs^2,1)), $
                              st_sr=hrbt_ssm+t70_midnight, $
                              relax=20.0, smooth=pp_runest_smooth) $
            else $
              havout = RunEst(maxorder=pp_maxorder_runest, $
                              tof=tof, stime=st, ctype=ctype, nc=nc, $
                              mc=mc, gdu=gdu, q=qual, $
                              bb_sr=sqrt(total(hrbv_scs^2,1)), $
                              st_sr=hrbt_ssm+t70_midnight, $
                              relax=20.0, sfac=pp_runest_sfac, $
                              smooth=pp_runest_smooth)
            
;            save, pp_maxorder_runest, tof, st, ctype, nc, mc, gdu, qual, $
;              hrbv_scs, hrbt_ssm, t70_midnight, pp_runest_smooth, havout, $
;              file='011009_c1_13-14_runest_out.idlsav'
;            stop
            
        endif else begin
            if (pp_runest_sfac eq 0) then $
              havout = RunEst(maxorder=pp_maxorder_runest, $
                              tof=tof, stime=st, ctype=ctype, nc=nc, $
                              mc=mc, gdu=gdu, q=qual, $
                              bb_sr=sqrt(bx^2+by^2+bz^2), st_sr=bt_t70, $
                              relax=20.0, smooth=pp_runest_smooth) $
            else $
              havout = RunEst(maxorder=pp_maxorder_runest, $
                              tof=tof, stime=st, ctype=ctype, nc=nc, $
                              mc=mc, gdu=gdu, q=qual, $
                              bb_sr=sqrt(bx^2+by^2+bz^2), st_sr=bt_t70, $
                              relax=20.0, sfac=pp_runest_sfac, $
                              smooth=pp_runest_smooth)
        endelse

        if (havout.status eq 0) then begin ; RunEst success for this hour
            
            dumstruc = {runorder:pp_bfill, $
                        tg:pp_rfill, $
                        prob:make_array(pp_maxorder_runest,/double,$
                                        value=pp_rfill), $
                        flag:make_array(pp_maxorder_runest,/int,$
                                        value=pp_bfill), $
                        estof:make_array(pp_maxorder_runest,/float,$
                                         value=pp_bfill)}
            runstruc1 = replicate( dumstruc, ntot ) ; DET1
            runstruc2 = replicate( dumstruc, ntot ) ; DET2
            ep_analyze_runest, id, id1, id2, havout, $
              tof, st, ctype, nc, mc, gdu, $
              runstruc1, runstruc2, stat=jstat
            if (not jstat) then begin
                printf, finfo.log_unit, $
                  'RunEst unsuccessful for this hour: Sortindex incompatible with input arrays - THIS SHOULD NOT HAPPEN!'
                printf, finfo.log_unit, $
                  'Skipping to next hour'
                printf, iff_unit, '%M '+ $
                  'RunEst unsuccessful for this hour: Sortindex incompatible with input arrays - THIS SHOULD NOT HAPPEN!'
                printf, iff_unit, '%M '+ $
                  'Skipping to next hour'
                goto, next_hour
            endif

        endif else begin        ; RunEst failure for this hour
            
            
;Missing Data Test=================
; 20010813, SC3, 11:00-12:00 and 12:00-13:00
; FGM data gap (~15 minutes in the spinres data) occurs near
; the 12:00 hour boundary
;                print, ihour*3600., (ihour+1)*3600.
;                print, hrbt_ssm(0), hrbt_ssm(n_elements(hrbt_ssm)-1)
;                stop
;Missing Data Test=================
            
            
            printf, finfo.log_unit, $
              'RunEst unsuccessful for this hour: '+havout.msg
            printf, finfo.log_unit, $
              'Skipping to next hour'
            printf, iff_unit, '%M '+ $
              'RunEst unsuccessful for this hour: '+havout.msg
            printf, iff_unit, '%M '+ $
              'Skipping to next hour'
            goto, next_hour
        endelse
    endif else begin            ; No q>=pp_qual_min and maxchan=7 data in whole hour
        printf, finfo.log_unit, $
          'RunEst unsuccessful for this hour: No q>=pp_qual_min and maxchan=7 data in whole hour'
        printf, finfo.log_unit, $
          'Skipping to next hour'
        printf, iff_unit, '%M '+ $
          'RunEst unsuccessful for this hour: No q>=pp_qual_min and maxchan=7 data in whole hour'
        printf, iff_unit, '%M '+ $
          'Skipping to next hour'
        goto, next_hour            
    endelse
    
; Analyze the data spin by spin
; Possible spins in this hour:  ~ 900      3600/4
; Possible beams in this hour:  ~ 231805   3600*330/5.125
    
    nspin = n_elements(bt_ssm)
    spin_width = mean(tspin,/double)/1d6 ; sec
    omega = 360d0/(spin_width)  ; degrees/sec
    spinstart_time_offset = pp_spinstart_angular_offset/omega ; sec
    
; Initialize the hourly iff-write variable
    if (pp_iff_hourly) then iff_output = ['']

    for ispin=0,nspin-1 do begin
        
        pstat.nspin_total(ihour) = pstat.nspin_total(ihour) + 1*pp_nchunk

; Define the left/right time boundaries for this spin interval
        lb_spin = bt_ssm(ispin) - spin_width/2d0
        rb_spin = bt_ssm(ispin) + spin_width/2d0
        lbd70 = lb_spin+t70_midnight
        lb0 = long(lbd70)
        lb1 = long( (lbd70 - lb0)*1d6 )
        pick_cvt_sec_vax, [lb0,lb1], lbstr_spin
        rbd70 = rb_spin+t70_midnight
        rb0 = long(rbd70)
        rb1 = long( (rbd70 - rb0)*1d6 )
        pick_cvt_sec_vax, [rb0,rb1], rbstr_spin
        
; Define the spin start time and the spin width
        spin_start_ssm_os = lb_spin
        spin_width_sec_os = spin_width
        
; Define the reconstituted SRT time, presumably the time that
; FGM uses to fix the inertial frame (i.e. to despin the data)
; for this spin interval. Recognize that this SRT time is NOT
; the spin start time...There is an official offset.
        spin_srt_ssm_fgm = spin_start_ssm_os - spinstart_time_offset
        spin_srt_t70_fgm = spin_srt_ssm_fgm + t70_midnight
        
; Find the PICK SRT time closest to and just prior to the spin start time.
; This time, in the best of all possible worlds, would exactly equal
; the reconsituted SRT time defined above (spin_srt_ssm_fgm).  Since
; Mark Chutter (PICK library) and Edita Georgescu (FGM co-I who
; provides the spin-averaged FGM data) don't use the same algorithms to
; define the sun reference pulse, and because the average tspin is
; used to define spin start time, these two times usually differ on the order
; of hundreths (or thousandths) of a second.
        ihit = where(srt_ssm lt spin_start_ssm_os)
        if (ihit(0) eq -1) then begin
;            printf, finfo.log_unit, '-----------------------------------------'
;            printf, finfo.log_unit, 'Start: '+lbstr_spin+'  Stop: '+rbstr_spin
;            printf, finfo.log_unit, '   No PICK SRT found to match FGM SRP'
            goto, next_spin
        endif
        ihit = ihit(n_elements(ihit)-1)
        spin_srt_ssm_pck = srt_ssm(ihit)
        spin_srt_t70_pck = srt_t70(ihit)

; Check that these two SRT's aren't too different
        if (abs(spin_srt_ssm_fgm-spin_srt_ssm_pck) gt .005) then begin    
;            printf, finfo.log_unit, '-----------------------------------------'
;            printf, finfo.log_unit, 'Start: '+lbstr_spin+'  Stop: '+rbstr_spin
;            printf, finfo.log_unit, '   PICK SRT and reconstituted FGM SRT times differ by more than .005 seconds: ',spin_srt_ssm_fgm-spin_srt_ssm_pck
;            printf, finfo.log_unit, '   This most likely indicates an EDI data gap'
            
; It's fatal if these two times are too different...
            goto, next_spin
        endif
        
; Choose which SRT (reconstituted or PICK) you want to use to de-spin
; the EDI data.  Note that the goal here is to use the one which is
; closest to what FGM uses, but since I can't get their exact time as
; output, I just usually use the one which gives the best "g dot
; B", which ... (has yet to be systematically determined)
        spin_srt_ssm_os = spin_srt_ssm_fgm
        
; Loop over the 'chunks' within this spin 
        
        dd = (rb_spin-lb_spin)/float(pp_nchunk)
        
        for ichunk=0,pp_nchunk-1 do begin
            
            lb = lb_spin + ichunk*dd
            rb = lb + dd
            chunk_mid_ssm = (lb+rb)/2.
            chunk_mid_t70 = t70_midnight + double(chunk_mid_ssm)
            
; Is this 'chunk' mid-time within the start and end times specified by
; the user?
            if (chunk_mid_ssm lt begt or $
                chunk_mid_ssm gt endt) then goto, next_chunk

            pstat.nspin_user(ihour) = pstat.nspin_user(ihour) + 1
            
; Is there any data within this chunk?
; NOTE: This particular filter was originally written using the atimes.
; Technically, it would be better to use the beam_times, but at this
; point in the code it would be difficult to implement since the
; beam_times are gun/det dependent whereas the atimes aren't. This
; just means that by using atimes instead of beam_times, only the
; beams located at the very edges of the chunk interval could be
; either erroneously included, or excluded from the calculation,
; something Goetz and I didn't think was too crucial.  The beam_times
; ARE USED FOR THE DE-SPINNING though, of course!!!
            
            id0 = where( atime_ssm ge lb and atime_ssm lt rb )
            if (id0(0) eq -1) then goto, next_chunk
            
; Define times for this 'chunk'
            lbd70 = lb+t70_midnight
            lb0 = long(lbd70)
            lb1 = long( (lbd70 - lb0)*1d6 )
            pick_cvt_sec_vax, [lb0,lb1], lbstr
            rbd70 = rb+t70_midnight
            rb0 = long(rbd70)
            rb1 = long( (rbd70 - rb0)*1d6 )
            pick_cvt_sec_vax, [rb0,rb1], rbstr
            mid70 = chunk_mid_t70
            mi0 = long(mid70)
            mi1 = long( (mid70 - mi0)*1d6 )
            pick_cvt_sec_vax, [mi0,mi1], mistr

;            print, 'Chunk Time: '+mistr
            
; Check the mode information which is within this
; chunk and output to the .IFF file an 'Instr_mode' line
; or an 'M' line if necessary
            bad = where(cnum(id0) ne 0)
            if (bad(0) ne -1) then begin
                for ibad=0,n_elements(bad)-1 do begin
                    inmod = inmodstr(id0(bad(ibad)))
                    chnum = cnum(id0(bad(ibad)))
                    otime70 = [atime0(id0(bad(ibad))),atime1(id0(bad(ibad)))]
                    otimessm = atime_ssm(id0(bad(ibad)))
                    if (chnum le 3 or chnum eq 10) then begin
                        if (inmod ne inmod_old) then begin
                            ep_write_iff_instmode, recnum, iff_unit, $
                              sc_number, inmod, pp_checkmsg_array(chnum), $
                              otime70, iff_hourly=pp_iff_hourly
                            edi_piso_write_iff_data, recnum, iff_unit, $
                              otimessm, rectime_old_ssm, /fill, iff_hourly=pp_iff_hourly
                            recnum = recnum + 1L
                            inmod_old = inmod
                        endif
                    endif else begin
                        ep_write_iff_checkmsg, iff_unit, inmod, $
                          pp_checkmsg_array(chnum), otime70, $
                          iff_hourly=pp_iff_hourly
                    endelse
                endfor
            endif
            
; Now make sure that the mode doesn't change within the chunk
; This is done AFTER the above Inst_mode check so that the 
; appropriate messages get written to the IFF file.               
            n_oc = n_elements(id0)
            mf_byte = bytarr(n_oc)
            for ifield = 0,7 do begin
                reads, strmid(inmodstr(id0),4+ifield*3,2), $
                  form='(z2.2)', mf_byte
                if (min(mf_byte) ne max(mf_byte)) then begin
                    ep_write_iff_checkmsg, iff_unit, 'NA', $
                      'More than 1 valid mode within this interval', $
                      [atime0(id0(0)),atime1(id0(0))], iff_hourly=pp_iff_hourly
                    goto, next_chunk
                endif
            endfor
            inmod_oc = inmodstr(id0(0))
            otime70 = [atime0(id0(0)),atime1(id0(0))]
            
; Define the average magnetic field vector for this chunk
            bx_scs = bx(ispin)
            by_scs = by(ispin)
            bz_scs = bz(ispin)
            bv1 = bvar1(ispin)
            bv2 = bvar2(ispin)
            if (use_hrfgm) then begin
                if (hrfgm_stat) then begin
                    ihr = where( hrbt_ssm ge lb and hrbt_ssm lt rb )
                    if (ihr(0) ne -1) then begin
                        
                        nhr = n_elements(ihr)
                        hrbt_ssm_c = hrbt_ssm(ihr)
                        hrbv_scs_c = hrbv_scs(0:2,ihr)
                        
                        bx_scs = total(hrbv_scs_c(0,*))/float(nhr)
                        by_scs = total(hrbv_scs_c(1,*))/float(nhr)
                        bz_scs = total(hrbv_scs_c(2,*))/float(nhr)
                        
; Normalised variance of the total magnetic field, bv1
; Normalised variance of the magnetic field magnitude, bv2
                        bmag = sqrt(total(hrbv_scs_c(0:2,*)^2,1))
                        bmagsqavg = total(bmag^2)/float(nhr)
                        bavgmagsq = bx_scs^2+by_scs^2+bz_scs^2
                        bmagavgsq = (total(bmag)/float(nhr))^2
                        bv1 = (bmagsqavg - bavgmagsq) / bmagsqavg
                        bv2 = (bmagsqavg - bmagavgsq) / bmagsqavg

                    endif else begin
                        message, 'HR FGM not available for this chunk', /cont
                        message, 'Using the spin-averaged field...', /cont
                    endelse
                endif else begin
                    message, 'HR FGM not available for this chunk', /cont
                    message, 'Using the spin-averaged field...', /cont
                endelse
            endif

; Get the rotation matrices at this point because scs2gse is needed
; when defining the 'BPP', or 'b-perp plane' coordinates in edi_piso_onechunk.pro.
; '09-Dec-2000 02:59:42.583'
            date = date2datevax(strmid(mistr,0,11),/reverse)
            year = long(strmid(date,0,4))
            mon = long(strmid(date,4,2))
            day = long(strmid(date,6,2))
            hour = long(strmid(mistr,12,2))
            minute = long(strmid(mistr,15,2))
            nlen = strlen(mistr)
            sec = double(strmid(mistr,18,nlen-18))
            att_stat = ep_extract_att(sc_number,chunk_mid_t70,decl,rasc,$
                                      pp_att_orb_hpath)
            if (not att_stat) then begin
                printf, log_unit, 'Attitude info extraction unsuccessful: '+$
                  mistr
                if (not pp_iff_hourly) then printf, iff_unit, $
                  '%M '+ 'Attitude info extraction unsuccessful: '+ mistr else $
                  iff_output=[iff_output,'%M '+ 'Attitude info extraction unsuccessful: '+ $
                              mistr]
                goto, next_chunk
            endif
            
            ep_get_haprotmat, year, mon, day, hour, minute, sec, $
              rasc, decl, ierr, $
              satgse=scs2gse, geigse=gei2gse
            if (ierr ne 0) then begin
                printf, log_unit, 'Haprotmat extraction unsuccessful: '+mistr
                if (not pp_iff_hourly) then printf, iff_unit, $
                  '%M '+ 'Haprotmat extraction unsuccessful: '+ mistr else $
                  iff_output=[iff_output,'%M '+ 'Haprotmat extraction unsuccessful: '+ $
                              mistr]
                goto, next_chunk
            endif
            
; Double pass required because there's no guarantee that all of the
; e_index values are the same (there's an energy switching mode)
; There is also now a pass over method (TRI and RMT) when pp_method=9,
; so we could have 4 results to choose from for any given 'chunk':
;saveit(0) : ieindex=0 (1 keV), ipass=0 
;saveit(1) : ieindex=0 (1 keV), ipass=1 
;saveit(2) : ieindex=1 (.5 keV), ipass=0 
;saveit(3) : ieindex=1 (.5 keV), ipass=1
; Just a counter matrix:
            cmat = make_array(2,2,/int)
            cmat(0,0) = 0
            cmat(0,1) = 1
            cmat(1,0) = 2
            cmat(1,1) = 3
; Reminder: If both energies successful, use the 1keV result
;           "Winner" will be written to the IFFALL file
;           "Winner" and "Loser" will be written to the EDI Archive
;                CDF (new)
;           pstat statistics are only collected for the "Winners"
            
            save_results_struc = {save_results_struc, $
                                  status:pp_bfill, $
                                  method:pp_bfill, $
                                  win_status:pp_bfill, $
                                  $
                                  date:finfo.date, $ ; Data field
                                  time_ssm:chunk_mid_ssm, $ ; Data field
                                  v_out:fltarr(3)+pp_rfill, $ ; Data field
                                  e_out:fltarr(3)+pp_rfill, $ ; Data field
                                  e_out2:fltarr(3)+pp_rfill, $ ; Data field
                                  e_out3:fltarr(3)+pp_rfill, $ ; Data field
                                  e_out4:fltarr(3)+pp_rfill, $ ; Data field
                                  rchi2:pp_rfill, $ ; Data field
                                  drift_step:pp_rfill, $ ; Data field
                                  d_error_inertial:pp_rfill, $ ; Data field
                                  phi_error_inertial:pp_rfill, $ ; Data field
                                  nbeam_usd:pp_i2fill, $ ; Data field
                                  out:intarr(384)+pp_bfill, $ ; Needed for pstat
                                  class:make_array(384,/string,value='F'), $ ; Needed for pstat
                                  used:intarr(384)+pp_bfill, $ ;Needed for pstat
                                  $
                                  $ ; Initialize sbytes with fill
                                  sbyte0:pp_bfill, $ ; S-byte 0
                                  sbyte3:pp_bfill, $ ; S-byte 3
                                  sbyte4:pp_bfill, $ ; S-byte 4
                                  sbyte5:pp_bfill, $ ; S-byte 5
                                  sbyte6:pp_bfill} ; S-byte 6

            saveit = replicate(save_results_struc, 4)

            nbeam_1000eV = 0L
            nbeam_500eV = 0L
            nbeam_tot = 0L
            nbeam_classA = 0L

            nbeam_1000eV_los = 0L
            nbeam_500eV_los = 0L
            nbeam_tot_los = 0L
            nbeam_classA_los = 0L

            for ieindex = 0,1 do begin
                
; Define which beams are in this interval and have energy index
; ieindex
                
                id = where( atime_ssm ge lb and atime_ssm lt rb and $
                            e_index eq ieindex and $
                            cnum eq 0 )
                
                if (id(0) eq -1) then goto, next_energy
                
; Apply the nbeam-minimum criteria and the maxchan criteria and the
; gunid criteria here as a quick way out...
; (this will be applied again in
; edi_piso_onechunk.pro; Here it's just a quick way out if the
; condition isn't met...)
                
; Gun2/Det1:
                id1 = where( (sq1(id) ge pp_qual_min) and $
                             (mxch1(id) ge pp_maxchan_min) and $
                             (mxch1(id) le pp_maxchan_max), c1)
                
; Gun1/Det2:
                id2 = where( (sq2(id) ge pp_qual_min) and $
                             (mxch2(id) ge pp_maxchan_min) and $
                             (mxch2(id) le pp_maxchan_max), c2)
                
                if (pp_gunid eq 0) then ctot=c1+c2 else $
                  if (pp_gunid eq 1) then ctot=c2 else ctot=c1
                
                if (ctot lt pp_nbeam_min) then goto, next_energy
                
; Define the one-chunk data to send to edi_piso_onechunk
                
                btime1_ssm_oc = btime1(id) - t70_midnight
                btime2_ssm_oc = btime2(id) - t70_midnight
                atime_ssm_oc = atime_ssm(id)
                atime_t70_oc = atime_t70(id)
                t1_oc        = t1(id)
                t1m2_oc      = t1m2(id)
                VAX1_oc      = VAX1(id)
                VAY1_oc      = VAY1(id)
                VAX2_oc      = VAX2(id)
                VAY2_oc      = VAY2(id)
                I1_oc        = I1(id)
                I2_oc        = I2(id)
                SQ1_oc       = SQ1(id)
                SQ2_oc       = SQ2(id)
                NNN_oc       = NNN(id)
                mm_oc        = mm(id)
                code_type_oc = code_type(id)
                mxch1_oc     = mxch1(id)
                mxch2_oc     = mxch2(id)
                runstruc1_oc = runstruc1(id)
                runstruc2_oc = runstruc2(id)
                
                n_oc = n_elements(id)
                bx_oc        = make_array(n_oc,/float,value=bx_scs)
                by_oc        = make_array(n_oc,/float,value=by_scs)
                bz_oc        = make_array(n_oc,/float,value=bz_scs)
                
                chunk_status = 0
                edi_piso_onechunk, time1, time1_2, bwidth1, data1, $
                  nnn1, mm1, ct1, $
                  maxchan1, gdotb1, runstat1, $
                  scs2bpp, bpp2scs, $
                  chunk_status, $
                  atime_t70_oc, $
                  spin_srt_t70_pck, spin_srt_t70_fgm, $
                  spin_srt_ssm_pck, spin_srt_ssm_fgm, $
                  unitout,unitout2, $
                  runstruc1_oc, runstruc2_oc, $
                  scs2gse=scs2gse, flip=flip, $
                  scs2bpp_refvec=scs2bpp_refvec
                
                if (chunk_status eq 0) then goto, next_energy
                
                nbeam = n_elements(data1(0,*))
                printf,finfo.log_unit, 'Start: '+lbstr+'  Stop: '+ $
                  rbstr+' Nbeams: '+strtrim(nbeam,2)
                
; adapted from the EDISER-Program of THB:
                
                plot_title = 'C'+sc_number+' / '+lbstr+'-'+ $
                  strmid(rbstr,12,15)
                
                nmchip = nnn1*mm1 ; This is correct because 'nnn' here 
                                ; is the parameter 'n' in all of the
                                ; documentation where the chip length
                                ; is defined.
                
                erg = pp_eindex2keV(ieindex)
                bmag_tesla = sqrt(bx_scs^2+by_scs^2+bz_scs^2)*1e-9
                bmag_nT = bmag_tesla/1e-9
                vb = sqrt(erg*2*1e3*pp_eV2joule/pp_emass_mks) ; meters/sec
                gyrorad = pp_emass_mks*vb/(pp_echarge_mks*bmag_tesla) ; m
                gyroper = 2.*!pi*gyrorad/vb*1.e-6 ; microseconds
                
;------------------------------------------------------------------
; whatech.pro is called only to redefine data1(1,*) (from 'qual' to
; 'out', as required by bestarg.pro) and to calculate
; rmax.  Define beam_qual before the call.
; It in no way decides which technique (TRI or TOF) will be used.
                beam_qual = fix(reform(data1(1,*)))
;                    whatech, data1, erg, nmchip, pp_sc_base, $
;                      rmax, yestof_whatech
                whatech_new, data1, erg, nmchip, pp_sc_base, $
                  rmax
                if (pp_rmax ne 0) then rmax=float(pp_rmax)
                
; Define the rest of the EDI beam information
                beam_gunid = fix(reform(data1(0,*)))
                beam_out = fix(reform(data1(1,*)))
                beam_xg = reform(data1(2,*))
                beam_yg = reform(data1(3,*))
                beam_alpha = reform(data1(4,*))
                beam_tof = reform(data1(5,*))
                beam_code_type = fix(reform(data1(6,*)))
                beam_atime = time1
                beam_btime = time1_2
                beam_bwidth = bwidth1
                beam_gdotb = gdotb1
                beam_runstat = runstat1
                beam_maxchan = maxchan1
                
                pstat.nspin_edata(ihour,ieindex) = $
                  pstat.nspin_edata(ihour,ieindex) + 1
                
; Bestarg ==========================================================
                ctref = systime(1)
                
                bestarg, nbeam, beam_gunid, beam_out, beam_xg, beam_yg, $
                  beam_alpha, beam_tof, beam_code_type, beam_qual, $
                  beam_bwidth, beam_runstat, $
                  beam_maxchan, beam_btime, $
                  erg, rmax, gyrorad, gyroper, flip, $
                  bmag=bmag_nT, $
                  keyplot3=pp_keyplot3, $
                  plot_title=plot_title, $
                  $
                  rmax_out=rmax_out, $
                  beam_class=beam_class, $
                  bclass_init=bclass_init, $
                  beam_penalty=beam_penalty, $
                  $
                  status_out=status_def, $
                  msg_out=msg_def, $
                  method_out=method_def, $
                  edi6_out=edi6_def, $
                  ambig_180_out=ambig_180_def, $
                  mean_angle_out=mean_angle_def, $
                  stdev_angle_out=stdev_angle_def, $
                  out_out=out_def, $
                  bestord_out=bestord_def, $
                  cat=classA_towards_def, $
                  nat=nonA_towards_def, $
                  caa=classA_away_def, $
                  naa=nonA_away_def, $
                  beam_used_out=beam_used_def, $
                  perr3_out=perr3_def, $
                  $
                  rchi2_CG=rchi2_CG, chi2_CG=chi2, ndegfree_CG=ndegfree, $
                  rchi2_unpen_CG=rchi2_unpen_CG, $
                  ndegfree_unpen_CG=ndegfree_unpen_CG, $
                  rchi2_FG=rchi2_FG, chi2_FG=chi2_FG, $
                  ndegfree_FG=ndegfree_FG, $
                  rchi2_unpen_FG=rchi2_unpen_FG, $
                  ndegfree_unpen_FG=ndegfree_unpen_FG, $
                  derr2=derr2, perr2=perr2, tgerr2=tgerr2, $
                  $
                  status_meth9 = status_meth9, $
                  msg_meth9 = msg_meth9, $
                  method_meth9 = method_meth9, $
                  edi6_meth9 = edi6_meth9, $
                  ambig_180_meth9 = ambig_180_meth9, $
                  mean_angle_meth9 = mean_angle_meth9, $
                  stdev_angle_meth9 = stdev_angle_meth9, $
                  out_meth9 = out_meth9, $
                  bestord_meth9 = bestord_meth9, $
                  classA_towards_meth9 = classA_towards_meth9, $
                  nonA_towards_meth9 = nonA_towards_meth9, $
                  classA_away_meth9 = classA_away_meth9, $
                  nonA_away_meth9 = nonA_away_meth9, $
                  beam_used_meth9 = beam_used_meth9, $
                  perr3_meth9 = perr3_meth9
                
                if (pp_method eq 9) then ip1 = 1 else ip1 = 0
                
                for ipass=0,ip1 do begin
                    
                    icount = cmat(ieindex,ipass)
                    
                    case ipass of
                        0:begin
                            status_out=status_def
                            msg_out=msg_def
                            method_out=method_def
                            edi6_out=edi6_def
                            ambig_180_out=ambig_180_def
                            mean_angle_out=mean_angle_def
                            stdev_angle_out=stdev_angle_def
                            out_out=out_def
                            bestord_out=bestord_def
                            classA_towards_out=classA_towards_def
                            nonA_towards_out=nonA_towards_def
                            classA_away_out=classA_away_def
                            nonA_away_out=nonA_away_def 
                            beam_used_out = beam_used_def
                            perr3_out = perr3_def
                            
                            if (pp_write_aux) then begin
; Must initialize these
                                b_gse = make_array(3,/float,value=pp_rfill)
                                tg = edi6_def(4)
                                tgerr = edi6_def(5)
                                d_scs = make_array(3,/float,value=pp_rfill)
                                v_scs = make_array(3,/float,value=pp_rfill)
                                e_scs = make_array(3,/float,value=pp_rfill)
                                d_gse = make_array(3,/float,value=pp_rfill)
;                                rchi2_CG
;                                rchi2_unpen_CG
                                clevel = pp_rfill
;                                rchi2_unpen_FG
                                clevel_FG = pp_rfill
;                                tgerr2
                                v_gse_nocorr = make_array(3,/float,value=pp_rfill)
                                e_gse_nocorr = make_array(3,/float,value=pp_rfill)
                                vsc_gse = make_array(3,/float,value=pp_rfill)
                            endif

                        end
                        1:begin
                            status_out = status_meth9
                            msg_out = msg_meth9
                            method_out = method_meth9
                            edi6_out = edi6_meth9
                            ambig_180_out = ambig_180_meth9
                            mean_angle_out = mean_angle_meth9
                            stdev_angle_out = stdev_angle_meth9
                            out_out = out_meth9
                            bestord_out = bestord_meth9
                            classA_towards_out = classA_towards_meth9
                            nonA_towards_out = nonA_towards_meth9
                            classA_away_out = classA_away_meth9
                            nonA_away_out = nonA_away_meth9
                            beam_used_out = beam_used_meth9
                            perr3_out = perr3_meth9

                            if (pp_write_aux) then begin
; Must initialize these
                                b_gse = make_array(3,/float,value=pp_rfill)
                                tg = edi6_meth9(4)
                                tgerr = edi6_meth9(5)
                                d_scs = make_array(3,/float,value=pp_rfill)
                                v_scs = make_array(3,/float,value=pp_rfill)
                                e_scs = make_array(3,/float,value=pp_rfill)
                                d_gse = make_array(3,/float,value=pp_rfill)
                                rchi2_CG = pp_rfill
                                rchi2_unpen_CG = pp_rfill
                                clevel = pp_rfill
                                rchi2_unpen_FG = pp_rfill
                                clevel_FG = pp_rfill
                                tgerr2 = pp_rfill
                                v_gse_nocorr = make_array(3,/float,value=pp_rfill)
                                e_gse_nocorr = make_array(3,/float,value=pp_rfill)
                                vsc_gse = make_array(3,/float,value=pp_rfill)
                            endif
                        end
                        
                    endcase
                    
                    ep_out = {status:pp_bfill, $
                              method:pp_bfill, $
                              ambig_180:pp_bfill, $
                              nbeam_usd:pp_i2fill, $
                              $
                              d:pp_rfill, $ ; Drift step magnitude [m]
                              derr:pp_rfill, $ ; Error on d [m]
                              derr2:pp_rfill, $ ; Error on d [m] using scaled chi-squared surface (use for TRI)
                              $
                              phi:pp_rfill, $ ; Drift azimuth [degrees,BPP]
                              perr:pp_rfill, $ ; Error on phi [degrees]
                              perr2:pp_rfill, $ ; Error on phi [degrees] using scaled chi-squared surface (use for TRI)
                              perr3:pp_rfill, $ ; Error on phi [degrees], recalculated (use for TOF)
                              $
                              out:$
                              make_array(nbeam,/int,value=pp_bfill), $
                              class:$
                              make_array(nbeam,/string,value='F'), $
                              used:$
                              make_array(nbeam,/int,value=pp_bfill), $
                              rchi2_FG:pp_rfill, $
                              magd_error_inertial:pp_rfill, $ ; meters
                              phid_error_inertial:pp_rfill, $ ; degrees
                              v_gse:make_array(3,/float,value=pp_rfill), $
                              e_gse:make_array(3,/float,value=pp_rfill), $
                              e_scs_earthframe: $
                              make_array(3,/float,value=pp_rfill), $
                              e_dsi_scframe: $
                              make_array(3,/float,value=pp_rfill), $
                              e_dsi_earthframe: $
                              make_array(3,/float,value=pp_rfill) $
                             }
                    
; Status flag tells what happened within bestarg.pro
                    ep_out.status = status_out
                    
; Things that are at least pre-defined (could even have information in
; them) even if the fit failed.  Whether or not they contain
; meaningful information depends on WHERE the fit failed.
                    ep_out.out = out_out
                    ep_out.class = beam_class
                    ep_out.used = beam_used_out
                    
                    if (status_out le 9) then begin ; Success
                        
                        ep_out.method = method_out
                        i = where(beam_used_out eq 1)
                        ep_out.nbeam_usd = n_elements(i)
                        
                        ep_out.d = edi6_out(0)
                        ep_out.derr = edi6_out(1)
                        ep_out.phi = edi6_out(2)
                        ep_out.perr = edi6_out(3)
                        ep_out.ambig_180 = ambig_180_out
                        
                        magd_error = ep_out.derr
                        phid_error = ep_out.perr
                        
                        clevel = pp_rfill
                        clevel_FG = pp_rfill
                        if (ep_out.method eq 0) then begin ; TRI only
                            ep_out.derr2 = derr2
                            ep_out.perr2 = perr2
                            ep_out.rchi2_FG = rchi2_FG
                            clevel = chi_sq_prob(chi2,ndegfree)
                            clevel_FG = chi_sq_prob(chi2_FG,ndegfree_FG)
                            magd_error = ep_out.derr2
                            phid_error = ep_out.perr2
                        endif else if (ep_out.method eq 1 or $
                                       ep_out.method eq 3) then begin ; PMT/RMT
                            ep_out.perr3 = perr3_out
                            phid_error = ep_out.perr3
                        endif
                        
; Calculate the 'd'erived 'p'arameters such as drift velocity and
; electric field from the best target information
                        
                        ep_calculate_dp, edi6_out, bpp2scs, scs2bpp, $
                          gei2gse, scs2gse,$
                          bx_scs, by_scs, bz_scs, $
                          sc_number, chunk_mid_t70, $
                          pp_att_orb_hpath, flip, finfo.log_unit, $
                          d_bpp, d_scs, d_gse, $ ; OUT
                          v_bpp, v_scs, v_gse, $ ; OUT
                          e_bpp, e_scs, e_gse, $ ; OUT
                          v_gse_nocorr=v_gse_nocorr, $ ; OUT
                          e_gse_nocorr=e_gse_nocorr, $ ; OUT
                          vsc_gse=vsc_gse, b_gse=b_gse, $ ; OUT
                          method=method_out, $ ; IN
                          e_scs_earthframe=e_scs_earthframe, $ ; OUT
                          magd_error=magd_error, phid_error=phid_error, $ ; IN
                          ambig_180=ambig_180_out, $ ; IN
                          mei=magd_error_inertial, pei=phid_error_inertial, $ ;OUT
                          e_dsi_scframe=e_dsi_scframe, $
                          e_dsi_earthframe=e_dsi_earthframe
                        
                        ep_out.e_dsi_scframe = e_dsi_scframe
                        ep_out.e_dsi_earthframe = e_dsi_earthframe
                        ep_out.e_scs_earthframe = e_scs_earthframe
                        ep_out.v_gse = v_gse
                        ep_out.e_gse = e_gse
                        ep_out.magd_error_inertial = magd_error_inertial
                        ep_out.phid_error_inertial = phid_error_inertial
                        
                    endif else begin
                        printf, finfo.log_unit, msg_out
                        if (not pp_iff_hourly) then printf, iff_unit, '%M '+msg_out $
                          else $
                          iff_output=[iff_output,'%M '+msg_out]
                    endelse
                    comptime = systime(1) - ctref
                    
; Save/Calculate/Determine everything you need for the IFF file for
; this energy
                    sout = saveit(icount)
                    ep_prep_iff_data_write, ep_out, sout, bv2, bmag_nT
                    saveit(icount) = sout
                    
;                    print, 'status_out:', status_out
                    if (write_diag_file) then begin
                        
                        printf, diag_unit, '==========================================================='
                        printf, diag_unit, '==========================================================='
                        
                        
                        printf, diag_unit, $
                          lbstr, rbstr, format = '(a27," - ",a27)'
                        
                        printf, diag_unit, msg_out
                        
                        printf, diag_unit, $
                          bx_scs, by_scs, bz_scs, bv1, bv2, $
                          format='("bx,by,bz,bv1,bv2: ",5(e12.5,1x))'
                        
                        mstr = ['TRI','NA','NA','RMT']
                        if (ep_out.method gt 3) then mmstr='FA' else mmstr=mstr(ep_out.method)
                        printf, diag_unit, $
                          ep_out.status, mmstr, comptime, $
                          format='("stat,meth,ctime: ",i3,1x,a3,1x,e13.6)'
                        
                        printf, diag_unit, $
                          ep_out.d, ep_out.derr, ep_out.derr2, $
                          format='("d,derr,derr2: ",3(e13.6,1x))'
                        
                        printf, diag_unit, $
                          ep_out.phi, ep_out.perr, ep_out.perr2, $
                          format='("phi,phierr,phierr2: ",3(e13.6,1x))'
                        
                        printf, diag_unit, $
                          rchi2_CG, ep_out.rchi2_FG, ep_out.ambig_180, $
                          format='("rchi2_CG,rchi2_FG,ambig: ",2(e13.6,1x),i1)'
                        
                        printf, diag_unit, $
                          mean_angle_out, stdev_angle_out, $
                          format='("mean_angle,stdev_angle: ",2(e13.6,1x))'
                        
                        printf, diag_unit, $
                          sout.sbyte0, 255, sout.sbyte3, $
                          sout.sbyte4, sout.sbyte5, sout.sbyte6, $
                          format='("sbytes: ",6(i3.3,2x))'
                        
                        qstr = ['BAD','CAUTION','GOOD']
                        if (sout.sbyte0 gt 2) then qqstr = 'FAIL' else qqstr =qstr(sout.sbyte0) 
                        printf, diag_unit, 'Data Quality: '+qqstr
                        
                        printf, diag_unit, 'ID MCh Qual Out Ord C0 C HAV_ord   HAV_flag      HAV_prob'                 
                        for ibeam = 0,nbeam-1 do begin
                            
                            pmr = string(pp_maxorder_runest,'(i1)')
                            printf, diag_unit, $
                              beam_gunid(ibeam), $
;	beam_xg(ibeam), $
;	beam_yg(ibeam), $
;	beam_alpha(ibeam), $
;	beam_tof(ibeam), $
;	beam_atime(ibeam), $
;	beam_btime(ibeam), $
;	beam_gdotb(ibeam), $
                            beam_maxchan(ibeam), $
;	beam_bwidth(ibeam)*!radeg, $
                            beam_qual(ibeam), $
                              ep_out.out(ibeam), $
                              bestord_out(ibeam), $
                              bclass_init(ibeam), $
                              beam_class(ibeam), $
                              beam_runstat(0,ibeam), $
                              beam_runstat(1:pp_maxorder_runest,ibeam), $
                              beam_runstat(2*pp_maxorder_runest+2:3*pp_maxorder_runest+1,ibeam), $
                              beam_penalty(ibeam), $
                              format= '(5(i1,3x),a1,2x,a1,4x,i1,2x,' + $
                              '"[",'+pmr+'(i1,1x),"] ",' + $ 
                              '"[",'+pmr+'(e10.3,1x),"]",1x,f5.2)'
                            
                        endfor
                        
                    endif
                    
; Output the auxilliary data
                    if (pp_write_aux) then begin
                        
                        printf, aux_unit, $
                          lbstr, rbstr, $
                          bx_scs, by_scs, bz_scs, bv1, bv2, $
                          nnn1, mm1, $
                          ct1, nbeam, erg, flip, $
                          b_gse, scs2bpp_refvec, $
                          rmax, rmax_out, $
                          format= $
                          '(2(a27,1x),'+$
                          '5(e13.6,1x),'+$
                          '2(i3,1x),'+$
                          'i1,1x,i3,1x,f3.1,1x,i1,1x,'+$
                          '3(e13.6,1x),3(e13.6,1x),e13.6,1x,e13.6)'
                        
                        nbstr = strtrim(nbeam,2)
                        ls = '('+nbstr
                        printf, aux_unit, beam_gunid, format = ls+'(i1,1x))'
                        printf, aux_unit, beam_xg, format = ls+'(e13.6,1x))'
                        printf, aux_unit, beam_yg, format = ls+'(e13.6,1x))'
                        printf, aux_unit, beam_alpha, format = ls+'(e13.6,1x))'
                        printf, aux_unit, beam_tof, format = ls+'(e13.6,1x))'
                        printf, aux_unit, beam_atime, format = ls+'(d20.13,1x))'
                        printf, aux_unit, beam_btime, format = ls+'(d20.13,1x))'
                        printf, aux_unit, beam_gdotb, format = ls+'(d20.13,1x))'
                        printf, aux_unit, beam_maxchan, format = ls+'(i2,1x))'
                        printf, aux_unit, beam_bwidth*!radeg, format=ls+'(f6.3,1x))'
                        printf, aux_unit, beam_qual, format = ls+'(i2,1x))'
                        
; THIS OLD FORMAT SOMETIMES WAS TOO BIG FOR THE OUTPUT BUFFER
;                    format= '('+ $
;                      nbstr+'(i1,1x),'+nbstr+'(e13.6,1x),'+nbstr+ $
;                      '(e13.6,1x),'+nbstr+'(e13.6,1x),'+nbstr+'(e13.6,1x),'+ $
;                      nbstr+'(d20.13,1x),'+nbstr+'(d20.13,1x),'+ $
;                      nbstr+'(e13.6,1x),'+nbstr+'(i2,1x))'
                        
                        printf, aux_unit, $
                          ep_out.status, ep_out.method, comptime, $
                          ep_out.d,ep_out.derr,ep_out.phi,ep_out.perr,tg,tgerr,$
                          d_scs, v_scs, e_scs, $
                          d_gse, ep_out.v_gse, ep_out.e_gse, $
                          rchi2_CG, rchi2_unpen_CG, clevel, $
                          ep_out.rchi2_FG, rchi2_unpen_FG, clevel_FG, $
                          ep_out.derr2, ep_out.perr2, tgerr2, $
                          v_gse_nocorr, e_gse_nocorr, $
                          vsc_gse, ep_out.ambig_180, $
                          mean_angle_out, stdev_angle_out, $
                          sout.sbyte0, 255, sout.sbyte3, $
                          sout.sbyte4, sout.sbyte5, sout.sbyte6, $
                          classA_towards_out, classA_away_out, $
                          nonA_towards_out, nonA_away_out, $
                          format=$
                          '(i3,1x,i3,1x,e13.6,1x,'+$
                          '6(e13.6,1x),'+$
                          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
                          '3(e13.6,1x),3(e13.6,1x),3(e13.6,1x),'+$
                          '3(e13.6,1x),'+$
                          '3(e13.6,1x),'+$
                          '3(e13.6,1x),'+$
                          '3(e13.6,1x),3(e13.6,1x),'+$
                          '3(e13.6,1x),i3,1x,'+$
                          '2(e13.6,1x),'+$
                          '3(i3,1x),'+$
                          '3(i3,1x),'+$
                          '2(i3,1x),'+$
                          '2(i3,1x))'
                        
                        printf, aux_unit, ep_out.out, $
                          format=ls+'(i3,1x))'
                        printf, aux_unit, bestord_out, $
                          format=ls+'(i3,1x))'
                        printf, aux_unit, bclass_init, $
                          format = ls+'(a1,1x))'
                        printf, aux_unit, beam_class, $
                          format = ls+'(a1,1x))'
                        printf, aux_unit, beam_penalty, $
                          format = ls+'(e13.6,1x))'
                        printf, aux_unit, beam_used_out, $
                          format=ls+'(i3,1x))'
                        
                    endif       ; pp_write_aux test
                    
                endfor          ; ipass counter

; Assign the "Winner" for this energy index
; Update the pstat variables
                ep_assign_winner, ieindex, saveit, iwin, ilos, ifail

                if (iwin ne -1) then begin

                    saveit(iwin).win_status = 1
                    
; Statistics needed for Status Bytes #1 and #2-------------------
; Winner:
                    n_used = n_elements(where(saveit(iwin).used eq 1))
                    n_classA = n_elements(where(saveit(iwin).used eq 1 and $
                                                saveit(iwin).class eq 'A'))
                    nbeam_tot = nbeam_tot + n_used
                    nbeam_classA = nbeam_classA + n_classA
                    
                    if (ieindex eq 0) then nbeam_1000eV = n_used $
                    else nbeam_500eV = n_used

; Loser:
                    if (ilos ne -1) then begin
                        n_used_los = n_elements(where(saveit(ilos).used eq 1))
                        n_classA_los = n_elements(where(saveit(ilos).used eq 1 and $
                                                        saveit(ilos).class eq 'A'))
                        nbeam_tot_los = nbeam_tot_los + n_used_los
                        nbeam_classA_los = nbeam_classA_los + n_classA_los
                        
                        if (ieindex eq 0) then nbeam_1000eV_los = n_used_los $
                        else nbeam_500eV_los = n_used_los
                    endif
;-----------------------------------------------------------------------
; Statistics needed for PSTAT file
; Only winner:
                    pstat.nspin_edata_succ(ihour,ieindex) = $
                      pstat.nspin_edata_succ(ihour,ieindex) + 1
                    
                    case saveit(iwin).method of
                        0:pstat.nspin_edata_succ_tri(ihour,ieindex)=$
                          pstat.nspin_edata_succ_tri(ihour,ieindex)+1
                        1:pstat.nspin_edata_succ_pmt(ihour,ieindex)=$
                          pstat.nspin_edata_succ_pmt(ihour,ieindex)+1
                        2:pstat.nspin_edata_succ_smt(ihour,ieindex)=$
                          pstat.nspin_edata_succ_smt(ihour,ieindex)+1
                        3:pstat.nspin_edata_succ_rmt(ihour,ieindex)=$
                          pstat.nspin_edata_succ_rmt(ihour,ieindex)+1
                    endcase
                    
                    pstat.nbeam_edata_succ(ihour,ieindex) = $
                      pstat.nbeam_edata_succ(ihour,ieindex) + $
                      n_elements(where(saveit(iwin).out ne pp_bfill))

                    d = where(saveit(iwin).class eq 'A',c)
                    pstat.nbeam_edata_succ_classA(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classA(ihour,ieindex)+c
                    d = where(saveit(iwin).class eq 'B',c)
                    pstat.nbeam_edata_succ_classB(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classB(ihour,ieindex)+c
                    d = where(saveit(iwin).class eq 'C',c)
                    pstat.nbeam_edata_succ_classC(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classC(ihour,ieindex)+c
                    d = where(saveit(iwin).class eq 'D',c)
                    pstat.nbeam_edata_succ_classD(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classD(ihour,ieindex)+c
                    d = where(saveit(iwin).class eq 'G',c)
                    pstat.nbeam_edata_succ_classG(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classG(ihour,ieindex)+c
                    d = where(saveit(iwin).class eq 'R',c)
                    pstat.nbeam_edata_succ_classR(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_classR(ihour,ieindex)+c

                    d = where(saveit(iwin).out eq 2,c)
                    pstat.nbeam_edata_succ_trioutlier(ihour,ieindex) = $
                      pstat.nbeam_edata_succ_trioutlier(ihour,ieindex)+c
                    
                    case saveit(iwin).sbyte0 of
                        0:pstat.nspin_edata_succ_bad(ihour,ieindex)=$
                          pstat.nspin_edata_succ_bad(ihour,ieindex)+1
                        1:pstat.nspin_edata_succ_caut(ihour,ieindex)=$
                          pstat.nspin_edata_succ_caut(ihour,ieindex)+1
                        2:pstat.nspin_edata_succ_good(ihour,ieindex)=$
                          pstat.nspin_edata_succ_good(ihour,ieindex)+1
                    endcase

                endif else begin

                    pstat.nspin_edata_fail(ihour,ieindex) = $
                      pstat.nspin_edata_fail(ihour,ieindex) + 1
                    pstat.nspin_edata_fail_stats(ihour,ieindex,$
                                                 saveit(ifail).status) = $
                      pstat.nspin_edata_fail_stats(ihour,ieindex,$
                                                   saveit(ifail).status) + 1
                endelse
                
                next_energy:
                
            endfor              ; ieindex counter
            
            if (nbeam_tot ne 0) then begin
                sbyte1 = long(float(nbeam_1000eV)/float(nbeam_tot)*100.)
                sbyte2 = long(float(nbeam_classA)/float(nbeam_tot)*100.)
            endif else begin
                sbyte1 = 255
                sbyte2 = 255
            endelse

            if (nbeam_tot_los ne 0) then begin
                sbyte1_los = long(float(nbeam_1000eV_los)/float(nbeam_tot_los)*100.)
                sbyte2_los = long(float(nbeam_classA_los)/float(nbeam_tot_los)*100.)
            endif else begin
                sbyte1_los = 255
                sbyte2_los = 255
            endelse
            
; Write data to IFF file now that the double-pass on energy index has
; been completed.
            
            if (inmod_oc ne inmod_old) then begin
                ep_write_iff_instmode, recnum, iff_unit, $
                  sc_number, inmod_oc, pp_checkmsg_array(0), otime70, $
                  iff_hourly=pp_iff_hourly
                inmod_old = inmod_oc
            endif
            
            edi_piso_write_iff_data, recnum, iff_unit, $
              saveit, rectime_old_ssm, sbyte1=sbyte1, sbyte2=sbyte2, $
              iff_hourly=pp_iff_hourly
            recnum = recnum + 1L

            if (pp_write_cdf) then begin
                ep_write_cdf_data, cdf_unit, date, sc_number, saveit, $
                  sbyte1, sbyte2, sbyte1_los, sbyte2_los
            endif
            
            if (pp_ediefw_out) then begin
                edi_piso_write_iff_data, recnum_ediefw, ediefw_unit, $
                  saveit, 0., sbyte1=sbyte1, sbyte2=sbyte2, $
                  /ediefw
                recnum_ediefw = recnum_ediefw + 1L
            endif

            if (pp_edidsi_out) then begin
                edi_piso_write_iff_data, recnum_edidsi, edidsi_unit, $
                  saveit, 0., sbyte1=sbyte1, sbyte2=sbyte2, $
                  /edidsi
                recnum_edidsi = recnum_edidsi + 1L
            endif

            if (pp_edidsi_inertial_out) then begin
                edi_piso_write_iff_data, recnum_edidsi_inertial, edidsi_inertial_unit, $
                  saveit, 0., sbyte1=sbyte1, sbyte2=sbyte2, $
                  /inertial_edidsi
                recnum_edidsi_inertial = recnum_edidsi_inertial + 1L
            endif
            
            next_chunk:
            
        endfor                  ; ichunk counter
        
        next_spin:
        
    endfor                      ; ispin counter

    if (pp_iff_hourly) then ep_write_iff_hourly, iff_unit
    
; Write out the PSTAT info every hour (in case of a crash, you don't
; want to have lost all of the prior information)
    ep_write_pstat_data, pstat_unit, ihour, pstat
    
    next_hour:
    
endfor                          ; ihour counter

test_out:

systime_end=systime(/seconds)

; Close log file in the wrapper, so that the profiling information
; will include that associated with edi_piso.pro
printf, finfo.log_unit, 'RUN END TIME: '+systime()
printf, finfo.log_unit, 'Calculation time (sec) : ',systime_end-systime_start
printf, finfo.log_unit, '%M Memory required: ', MEMORY(/HIGHWATER) - start_mem

; Close iff file
; But first, write a fill record if no data records have been written yet.
if (recnum eq 0) then begin
    edi_piso_write_iff_data, recnum, iff_unit, 0., 0., /fill
    recnum = recnum + 1L
endif
printf, iff_unit, 'E '+strtrim(recnum,2)
printf, iff_unit, '%M Calculation time (sec) : ', systime_end-systime_start
printf, iff_unit, '%M '+finfo.iff_name+' closed at '+systime()
printf, iff_unit, '%M Memory required: ', MEMORY(/HIGHWATER) - start_mem

free_lun, iff_unit

; Close auxilliary files
if (pp_write_aux) then free_lun, aux_unit
if (pp_write_cdf) then cdf_close, cdf_unit
if (pp_ediefw_out) then begin
    if (recnum_ediefw eq 0) then begin
        edi_piso_write_iff_data, recnum_ediefw, ediefw_unit, 0., 0., /fill
        recnum_ediefw = recnum_ediefw + 1L
    endif
    printf, ediefw_unit, 'E '+strtrim(recnum_ediefw,2)   
    free_lun, ediefw_unit
endif

if (pp_edidsi_out) then begin
    if (recnum_edidsi eq 0) then begin
        edi_piso_write_iff_data, recnum_edidsi, edidsi_unit, 0., 0., /fill
        recnum_edidsi = recnum_edidsi + 1L
    endif
    printf, edidsi_unit, 'E '+strtrim(recnum_edidsi,2)   
    free_lun, edidsi_unit
endif

if (pp_edidsi_inertial_out) then begin
    if (recnum_edidsi_inertial eq 0) then begin
        edi_piso_write_iff_data, recnum_edidsi_inertial, edidsi_inertial_unit, 0., 0., /fill
        recnum_edidsi_inertial = recnum_edidsi_inertial + 1L
    endif
    printf, edidsi_inertial_unit, 'E '+strtrim(recnum_edidsi_inertial,2)   
    free_lun, edidsi_inertial_unit
endif

free_lun, pstat_unit

if (write_diag_file) then free_lun, diag_unit

return
END
