; = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
;+
; NAME:
;     GET_FIELDINFO
;
; IDENT:
;     $Id: get_fieldinfo_hdf.pro,v 1.1 2003/02/14 20:17:45 glennh Exp $
;
; PURPOSE:
;     determine the dimensions and types of HDF file field thingies
;
; AUTHOR:
;     Ed Santiago
;
; CALLING SEQUENCE:
;     x = GET_FIELDINFO( filename, basename, vdata, fields )
;
; INPUTS:
;     filename     - name of the HDF input file
;     basename     - name of the ASC thingy (eg, "swepam_i")
;     vdata        - HDF vdata thingy
;     fields       - comma-separated string of fields to be extracted
;
; OUTPUTS:
;     an array of structures
;
; SIDE EFFECTS:
;
; EXAMPLE:
;      fid   = HDF_Open(files[i])
;      v     = HDF_VD_Find(fid,'swepam_i')
;      vdata = HDF_VD_Attach(fid, v)
;      fields = 'foo,bar'
;
;      fieldinfo   = get_fieldinfo(files[i], 'swepam_i', vdata, fields)
;
;-
FUNCTION get_fieldinfo, filename, basename, vdata, fields
  ; define the "fieldinfo" structure, containing field name/type/dimensions
  fieldinfo = { fieldinfo, name:'', type:'', ndims:0L, dims:LonArr(8) }
  fieldlist = [ fieldinfo ]

  ; Yuk.  IDL is case-insensitive, but HDFs are not.  The fields are
  ; stored in case-sensitive manner, so if we are called with "bmag"
  ; and the field is actually called "Bmag", we will fail.
  ;
  ; To avoid this, we always read in all the VD and SD field names,
  ; then do a case-insensitive comparison of the fields passed in.
  ; This allows us to find out the *real* HDF name of the field.

  ; obtain the VDATA fields
  HDF_VD_Get, vdata, fields=fields_all

  ; obtain the SDATA fields
  sd = HDF_SD_Start(filename)
  HDF_SD_FileInfo, sd, datasets, attributes
  FOR i=0,datasets-1 DO BEGIN
      sds = HDF_SD_Select(sd,i)
      HDF_SD_GetInfo, sds, name=nnn
      IF STRMID(nnn,0,STRLEN(basename)+1) EQ basename + '_' THEN BEGIN
          fields_all = fields_all + ',' + STRMID(nnn, STRLEN(basename)+1, 999)
      ENDIF
      HDF_SD_EndAccess, sds
  ENDFOR
  HDF_SD_End, sd


  ; If no fields were given, assume user wants them all
  IF N_Elements(fields) EQ 0 THEN fields = fields_all

  ;
  ; Split the list of fields into an array.  For each element, do some 
  ; HDF info-get stuff to figure out what it is.
  ;
  fieldarr     = split(',', fields)
  fieldarr_all = split(',', fields_all)
  FOR i = 0, N_Elements(fieldarr)-1 DO BEGIN
      ; Find out the real (HDF case-sensitive) name of the field
      tmp = where(StrUpCase(fieldarr_all) EQ StrUpCase(fieldarr[i]), c)
      IF c NE 1 THEN BEGIN
        MESSAGE, 'no such VD or SD: ' + fieldarr[i], /Continue
        GOTO, skipThisField
      ENDIF
      fieldinfo.name  = fieldarr_all[tmp]

      ; HDF routines choke on errors, but we want to proceed.
      Catch, err_sts

      IF err_sts EQ 0 THEN BEGIN
          ;
          ; First, see if it's a VDATA.  If it is, life is simple.  If
          ; it isn't, the following line will choke, and control will 
          ; pass on to the ELSE below.
          ;
          HDF_VD_GetInfo, vdata, fieldinfo.name, Type=t, Order=o
          IF o EQ 1 THEN fieldinfo.ndims = 0 ELSE fieldinfo.ndims = 1
          fieldinfo.dims[0] = o
      ENDIF ELSE IF !Error_State.Name EQ 'IDL_M_HDF_VS_FEXIST' THEN BEGIN
	  ;
	  ; Sigh, the HDF_VD_GetInfo call failed.  That means that the
	  ; field in question is not a VDATA.  Now try to see if it's 
	  ; an SDATA.  Since this HDF routine chokes also, we need a
	  ; new CATCH.
	  ;
          Catch, err_sts
          IF err_sts EQ 0 THEN BEGIN
	      ;
	      ; Initialize the SDATA interface, and try to get info.  If
	      ; it works, we set the dimensions and type.
	      ;
              sd  = HDF_SD_Start(filename)
              sdi = HDF_SD_NameToIndex(sd, basename + '_' + fieldinfo.name)
              sds = HDF_SD_Select(sd, sdi)

              HDF_SD_GetInfo, sds, Type=t, Dims=o
              fieldinfo.ndims = N_Elements(o) - 1
              fieldinfo.dims  = o
	      HDF_SD_EndAccess, sds
	      HDF_SD_End, sd
          ENDIF ELSE IF !Error_State.Name EQ 'IDL_M_HDF_SD_SELECT' THEN BEGIN
	      ; Nope, could not find a VDATA or SDATA by that name.
              print, '['+filename+': no such VD or SD: ' + fieldarr[i] + ']'
          ENDIF ELSE BEGIN
	      ; Weird unexpected error
              print, 'hdf_sd failed, err=',err_sts
              stop
          ENDELSE
      ENDIF ELSE BEGIN
	  ; Weird unexpected error
	  Catch, /Cancel
          print, 'hdf_vd_info failed with status ', err_sts, fieldarr[i]
	  Message, !ERR_STRING, /NoName
      ENDELSE
      Catch, /Cancel

      IF err_sts EQ 0 THEN BEGIN
          fieldinfo.type  = t
          fieldlist = [ fieldlist, fieldinfo ]
      ENDIF
skipThisField:
  ENDFOR	; i=0,nfields

  RETURN, fieldlist[1:*]
END

